package org.openmetadata.util.xmlbeans;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

import org.apache.xmlbeans.SchemaType;
import org.apache.xmlbeans.SchemaTypeLoader;
import org.apache.xmlbeans.XmlBeans;
import org.apache.xmlbeans.XmlException;
import org.apache.xmlbeans.XmlObject;
import org.apache.xmlbeans.XmlOptions;

public class XmlObjectCaster {

	private final static XmlOptions fragmentLoadOptions = new XmlOptions()
			.setLoadReplaceDocumentElement(null);

	/**
	 * Will tell you whether the provided <code>source</code> object can be cast
	 * to the provided <code>targetClass</code>. You can specify whether the
	 * conversion should allow up-casting (i.e. converting an object to a
	 * derived type from a base type). When up-casting is performed, if the
	 * target is an extension there may be missing content; if the target is a
	 * restriction there maybe invalid extra content.
	 * 
	 * @param targetClass
	 *            to class to which the <code>source</code> is to be converted
	 * @param source
	 *            the object to be cast
	 * @param allowUpCasting
	 *            <code>true</code> if conversion to a type derived from the
	 *            source type should be allowed
	 * @return <code>true</code> if the <code>source</code> object can be cast
	 *         to the provided <code>targetClass</code>, otherwise
	 *         <code>false</code>
	 */
	public static final boolean canCast(Class<? extends XmlObject> targetClass,
			XmlObject source, boolean allowUpCasting) {
		SchemaTypeLoader loader = XmlBeans.getContextTypeLoader();
		SchemaType targetType = XmlBeans.typeForClass(targetClass);
		if (targetType.equals(XmlObject.type)) {
			// If the request is simply for XmlObject, cast and return it
			return true;
		}
		SchemaType sourceType = source.schemaType();
		if (sourceType.isDocumentType()) {
			// Compare to target type
			if (targetType.isDocumentType()) {
				// ***Document to Document conversion***
				if (targetType.equals(sourceType)) {
					// These object is correct and requires no casting
					return true;
				} else if (targetType.isValidSubstitution(sourceType
						.getDocumentElementName())) {
					// The source if a valid substitution for the target
					// therefore the source content type must be validly derived
					return true;
				} else if (sourceType.isValidSubstitution(targetType
						.getDocumentElementName())) {
					// The target is a valid substitution group for the source,
					// so this can be changed but doing so requires up-casting
					return allowUpCasting;
				}
			} else {
				// ***Document to Type conversion***
				// Determine the base type of the content for the source object
				// (this may not be the actual type in the source object)
				SchemaType baseContentType = loader.findElement(
						sourceType.getDocumentElementName()).getType();
				// Get the actual content and determine its actual type
				XmlObject sourceContent = getDocumentContent(source,
						baseContentType, false);
				SchemaType sourceContentType = sourceContent.schemaType();
				if (targetType.isAssignableFrom(sourceContentType)) {
					// Source content type is derived (or the same) as the
					// target type so can be cast as is
					return true;
				} else if (sourceContentType.isAssignableFrom(targetType)) {
					// Target type is derived from the source content type so
					// up-casting will be necessary
					return allowUpCasting;
				}
			}
		} else {
			if (targetType.isDocumentType()) {
				// ***Type to Document conversion***
				// Get the base type for the request document so we can check it
				// against the source type
				targetType = loader.findElement(
						targetType.getDocumentElementName()).getType();
			}
			// ***Type to Type conversion***
			if (sourceType.equals(XmlBeans.NO_TYPE)) {
				// The source can be cast to the target, but it may not be
				// valid since we have no idea of its actual type
				return allowUpCasting;
			} else if (targetType.isAssignableFrom(sourceType)) {
				// Source type is derived (or the same) as the target type
				// so can be cast as is
				return true;
			} else if (sourceType.isAssignableFrom(targetType)) {
				// Target type is derived from the source type so up-casting
				// will be necessary
				return allowUpCasting;
			}
		}
		return false;
	}

	/**
	 * Will attempt to cast the <code>source XmlObject</code> to the
	 * <code>targetClass</code>. Users should call the
	 * {@link #canCast(Class, XmlObject, boolean)} method to ensure the cast can
	 * be performed before calling this method.
	 * <p>
	 * This method will perform the casting, even if up-casting is required.
	 * When up-casting is performed, if the target is an extension there may be
	 * missing content; if the target is a restriction there maybe invalid extra
	 * content.
	 * 
	 * @param targetClass
	 *            to class to which the <code>source</code> is to be converted
	 * @param source
	 *            the object to be cast
	 * @return the <code>source</code> object strongly typed as the
	 *         <code>targetClass</code>
	 */
	@SuppressWarnings("unchecked")
	public static final <S extends XmlObject> S cast(Class<S> targetClass,
			XmlObject source) {
		SchemaTypeLoader loader = XmlBeans.getContextTypeLoader();
		SchemaType targetType = XmlBeans.typeForClass(targetClass);
		SchemaType sourceType = source.schemaType();
		if (targetType.equals(sourceType)) {
			// The types are the same so just return the object
			return targetClass.cast(source);
		}
		if (targetType.equals(XmlObject.type)) {
			// If the request is simply for XmlObject, cast and return it
			return targetClass.cast(source);
		}
		if (sourceType.isDocumentType()) {
			// Compare to target type
			if (targetType.isDocumentType()) {
				// ***Document to Document conversion***
				// Get the base source content type (this may not be the actual
				// content type)
				SchemaType sourceContentType = loader.findElement(
						sourceType.getDocumentElementName()).getType();

				// Get the content type for the target document
				SchemaType targetContentType = loader.findElement(
						targetType.getDocumentElementName()).getType();
				if (targetType.equals(sourceType)) {
					// These object is correct so "cast" and return
					return targetClass.cast(source);
				} else if (targetType.isValidSubstitution(sourceType
						.getDocumentElementName())) {
					// The source if a valid substitution for the target
					// therefore the source content type must be validly derived

					// Instantiate target empty document
					S request = targetClass.cast(loader.newInstance(targetType,
							null));
					// Add new content to the empty target document and
					// set the source content to the new target content
					// NOTE: this will result in an xsi:type if the source
					// content type is different from the target content type,
					// but since it must be validly derived it will be valid
					getDocumentContent(request, targetContentType, true)
							.set(getDocumentContent(source, sourceContentType,
									false));
					return request;
				} else if (sourceType.isValidSubstitution(targetType
						.getDocumentElementName())) {
					// The target is a valid substitution group for the source,
					// so this can be changed but doing so requires up-casting
					// Get the source content and create a substitution to the
					// target document
					XmlObject sourceContent = getDocumentContent(source,
							sourceContentType, false).substitute(
							targetType.getDocumentElementName(),
							targetContentType);
					// Instantiate a new empty target document
					S request = targetClass.cast(XmlBeans
							.getContextTypeLoader().newInstance(targetType,
									null));
					// Add new content to the empty target document and set the
					// source content which was substituted above to this.
					// NOTE: the type will match but may be invalid due to
					// up-casting
					getDocumentContent(request, targetContentType, true).set(
							sourceContent);
					return request;
				}
			} else {
				// ***Document to Type conversion***
				// Determine the base type of the content for the source object
				// (this may not be the actual type in the source object)
				SchemaType baseContentType = loader.findElement(
						sourceType.getDocumentElementName()).getType();
				// Get the actual content and determine its actual type
				XmlObject sourceContent = getDocumentContent(source,
						baseContentType, false);
				SchemaType sourceContentType = sourceContent.schemaType();
				if (targetType.isAssignableFrom(sourceContentType)
						&& !sourceType.equals(XmlBeans.NO_TYPE)) {
					// Source content type is derived (or the same) as the
					// target type so can be cast as is
					return targetClass.cast(sourceContent);
				} else if (sourceContentType.isAssignableFrom(targetType)) {
					// Target type is derived from the source content type so
					// up-casting will be necessary and result may not be valid
					return targetClass.cast(sourceContent
							.changeType(targetType));
				}
			}
		} else {
			if (targetType.isDocumentType()) {
				// ***Type to Document conversion***
				// Get the base type for the request document
				SchemaType requestContentType = loader.findElement(
						targetType.getDocumentElementName()).getType();
				if (requestContentType.isAssignableFrom(sourceType)
						|| sourceType.equals(XmlBeans.NO_TYPE)) {
					// Source is derived (or the same) as the target content
					// type so it can be used as the content for the document as
					// is (will result in xsi:type if the source type is
					// different).

					// Instantiate an empty target document
					S target = targetClass.cast(loader.newInstance(targetType,
							null));
					// Add the target content and set the source content to it
					// NOTE: this will result in an xsi:type if the source
					// content type is different from the target content type,
					// but it will be valid
					getDocumentContent(target, requestContentType, true).set(
							cast(requestContentType.getJavaClass(), source));
					return target;
				} else if (sourceType.isAssignableFrom(requestContentType)) {
					// Target content type is derived from the source type so
					// up-casting will be necessary

					// Instantiate an empty target document
					S target = targetClass.cast(loader.newInstance(targetType,
							null));
					// Add the target content and set its content to the
					// up-casted source
					// NOTE: this may not be valid
					getDocumentContent(target, requestContentType, true).set(
							source.changeType(requestContentType));
					return target;
				}
			} else {
				// ***Type to Type conversion***
				if (sourceType.equals(XmlBeans.NO_TYPE)) {
					// Source does not have a type so we have to parse it as a
					// fragment
					try {
						return targetClass.cast(loader.parse(
								source.newXMLStreamReader(), targetType,
								fragmentLoadOptions));
					} catch (XmlException e) {
						throw new RuntimeException(e);
					}
				} else if (targetType.isAssignableFrom(sourceType)) {
					// Source type is derived (or the same) as the target type
					// so can be cast as is
					return targetClass.cast(source);
				} else if (sourceType.isAssignableFrom(targetType)) {
					// Target type is derived from the source type so up-casting
					// will be necessary and the target may not be valid.
					return targetClass.cast(source.changeType(targetType));
				}
			}
		}
		throw new RuntimeException(
				"Requested document type is not valid for serialization type.");
	}

	private static XmlObject getDocumentContent(XmlObject document,
			SchemaType contentType, boolean isNew) {
		String nameStart = isNew ? "addNew" : "get";
		for (Method m : document.getClass().getMethods()) {
			if (m.getName().startsWith(nameStart)
					&& m.getReturnType().equals(contentType.getJavaClass())) {
				try {
					XmlObject xo = (XmlObject) m.invoke(document);
					if (xo == null && !isNew) {
						return getDocumentContent(document, contentType, true);
					}
					return xo;
				} catch (IllegalArgumentException e) {
					e.printStackTrace();
				} catch (IllegalAccessException e) {
					e.printStackTrace();
				} catch (InvocationTargetException e) {
					e.printStackTrace();
				}
			}
		}
		throw new RuntimeException("Could not find document content getter.");
	}

}
