package org.openmetadata.util.xmlbeans;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Set;

import org.apache.commons.lang3.RandomStringUtils;
import org.apache.xmlbeans.XmlCursor;
import org.apache.xmlbeans.XmlCursor.TokenType;
import org.apache.xmlbeans.XmlID;
import org.apache.xmlbeans.XmlIDREF;
import org.apache.xmlbeans.XmlIDREFS;
import org.apache.xmlbeans.XmlObject;

/**
 * A utility that will randomize XML ids in an XML document.
 * 
 * @author Jack Gager (j.gager@mtna.us)
 *
 */
public class XmlIdRandomizer {

	public static int ID_LENGHT = 6;

	/**
	 * Updates and XML object by randomizing the XML ids in the document,
	 * ensuring that they do not clash with the provided set of ids. All
	 * references to the ids are updated as well. Note that this only works with
	 * ids and references contained in attributes.
	 * 
	 * @param xml
	 *            the xml object to update; this will be directly updated
	 * @param usedIds
	 *            the set of ids used; this will be directly updated with the
	 *            new ids created
	 */
	public static void randomizeIds(XmlObject xml, Set<String> usedIds) {
		HashMap<String, String> idMap = new HashMap<>();

		// randomize the ids
		XmlCursor cursor = xml.newCursor();
		while (cursor.hasNextToken()) {
			if (cursor.toNextToken() == TokenType.ATTR) {
				XmlObject xo = cursor.getObject();
				if (xo instanceof XmlID) {
					XmlID idAtt = (XmlID) xo;
					String id = generateId(usedIds);
					idMap.put(idAtt.getStringValue(), id);
					idAtt.setStringValue(id);
				}
			}
		}

		// replace the id references
		cursor = xml.newCursor();
		while (cursor.hasNextToken()) {
			if (cursor.toNextToken() == TokenType.ATTR) {
				XmlObject xo = cursor.getObject();
				if (xo instanceof XmlIDREF) {
					((XmlIDREF) xo).setStringValue(idMap.get(((XmlIDREF) xo).getStringValue()));
				} else if (xo instanceof XmlIDREFS) {
					ArrayList<String> ids = new ArrayList<>();
					for (Object id : ((XmlIDREFS) xo).getListValue()) {
						ids.add(idMap.get((String) id));
					}
					((XmlIDREFS) xo).setListValue(ids);
				}
			}
		}
	}

	private static String generateId(Set<String> usedIds) {
		String id;
		while (!usedIds.add((id = RandomStringUtils.randomAlphanumeric(ID_LENGHT))))
			;
		return id;
	}

}
