package org.openmetadata.util.xmlbeans;

import java.util.ArrayList;

import javax.xml.namespace.QName;

import org.apache.xmlbeans.QNameSet;
import org.apache.xmlbeans.XmlCursor;
import org.apache.xmlbeans.XmlObject;

/**
 * This utility class contains static methods for setting and getting attributes
 * from an XML Object and is intended to be used when the XML schema allows for
 * any attributes to be present so that the resulting XMLBean for the type will
 * not have explicit getters and setters.
 * 
 * @author Jack Gager
 * 
 */
// TODO The setting methods might check for validity such as an existing
// attribute or a clash in namespace prefixes
public class AnyAttributeUtilities {

	/**
	 * A convenience method for {@link #addAttribute(XmlObject, QName, String)}
	 * for adding an qualified attribute, with a pre-assigned prefix.
	 * 
	 * @param xo
	 *            the <code>XmlObject</code> to which the attribute will be
	 *            added
	 * @param attributeName
	 *            the name for the attribute
	 * @param attributeNamespace
	 *            the namespace URI for the attribute
	 * @param attributePrefix
	 *            the prefix to assign to the attribute namespace URI
	 * @param attributeValue
	 *            the attribute value
	 */
	public static void addAttribute(XmlObject xo, String attributeName,
			String attributeNamespace, String attributePrefix,
			String attributeValue) {
		addAttribute(xo, new QName(attributeNamespace, attributeName,
				attributePrefix), attributeValue);
	}

	/**
	 * A convenience method for {@link #addAttribute(XmlObject, QName, String)}
	 * for adding an qualified attribute, without a pre-assigned prefix.
	 * 
	 * @param xo
	 *            the <code>XmlObject</code> to which the attribute will be
	 *            added
	 * @param attributeName
	 *            the name for the attribute
	 * @param attributeNamespace
	 *            the namespace URI for the attribute
	 * @param attributeValue
	 *            the attribute value
	 */
	public static void addAttribute(XmlObject xo, String attributeName,
			String attributeNamespace, String attributeValue) {
		addAttribute(xo, new QName(attributeNamespace, attributeName),
				attributeValue);
	}

	/**
	 * A convenience method for {@link #addAttribute(XmlObject, QName, String)}
	 * for adding an unqualified (local) attribute.
	 * 
	 * @param xo
	 *            the <code>XmlObject</code> to which the attribute will be
	 *            added
	 * @param attributeName
	 *            the name for the local attribute
	 * @param attributeValue
	 *            the attribute value
	 */
	public static void addAttribute(XmlObject xo, String attributeName,
			String attributeValue) {
		addAttribute(xo, new QName(attributeName), attributeValue);
	}

	/**
	 * Adds an attribute with a value to the supplied <code>XmlObject</code>
	 * 
	 * @param xo
	 *            the <code>XmlObject</code> to which the attribute will be
	 *            added
	 * @param attributeName
	 *            the {@link QName} for the new attribute
	 * @param attributeValue
	 *            the attribute value
	 */
	public static void addAttribute(XmlObject xo, QName attributeName,
			String attributeValue) {
		XmlCursor cursor = xo.newCursor();
		cursor.toFirstContentToken();
		cursor.insertAttributeWithValue(attributeName, attributeValue);
		cursor.dispose();
	}

}
