package org.openmetadata.util.xmlbeans;

import java.util.ArrayList;
import java.util.Arrays;

import org.apache.log4j.Logger;
import org.apache.xmlbeans.XmlCursor;
import org.apache.xmlbeans.XmlObject;
import org.apache.xmlbeans.XmlRuntimeException;
import org.openmetadata.util.xmlbeans.ValidatorExpression.LOG_LEVEL;
import org.openmetadata.util.xmlbeans.ValidatorExpression.TYPE;

/**
 * Performs second level validation on an Apache Bean XmlObject 
 * by applying a collection of xpath statements or xqueries
 * and reporting error/warning/info in the standard logger
 * based on the return value.
 * 
 * For validation purposes (warning/error entries), the xpath/xquery 
 * is expected to return a boolean true/false.
 * Other values would result in an error.
 * 
 * For reporting purposes (info entries), the results will be reported as text.
 * 
 * The validator is configured by providing a collection of 
 * objects containing the xpath/xquery expression  and their 
 * corresponding reporting level.These can be fed as objects
 * or string (see method below for formatting rules)
 * 
 *  To facilitate the xpath/xquery, it is also possible to 
 *  specify namespace prefixes that are included in all 
 *  xpath/xquery statements.
 * 
 * @author Pascal Heus (pascal.heus@gmail.com)
 *
 */
public class Validator {
	Logger logger = Logger.getLogger(this.getClass().getName());
	
	XmlObject xmlObject;
	ArrayList<ValidatorExpression> expressions = new ArrayList<ValidatorExpression>();
	String separator="\\|";
	
	//===============================================================
	// CONSTRUCTOR(S)
	//===============================================================
	public Validator() {
	}
	
	public Validator(XmlObject xmlObject) {
		setXmlObject(xmlObject);
	}

	//===============================================================
	// METHODS
	//===============================================================
	
	public void addExpressions(ValidatorExpression... expressions) {
		this.expressions.addAll(Arrays.asList(expressions));
	}
	
	/**
	 * Adds a collection of string based expressions 
	 * using default field separator
	 *  
	 * @param strExpressions
	 */
	public void addExpressions(String... strExpressions) {
		addExpressions(this.separator,strExpressions);
	}
	
	/**
	 * Adds a single string based expression 
	 * using default field separator
	 * 
	 * @param strExpression
	 */
	public void addExpression(String strExpression) {
		addExpression(this.separator,strExpression);
	}
	
	/**
	 * Adds a collection of string based expressions
	 * using specified field separator 
	 * 
	 * @param separator
	 * @param strExpressions
	 */
	public void addExpressions(String separator, String... strExpressions) {
		for(String strExpression : strExpressions) {
			addExpression(separator,strExpression);
		}
	}

	/**
	 * Adds a single string based expression 
	 * using specified field separator 
	 * 
	 * @param separator
	 * @param strExpression
	 */
	public void addExpression(String separator, String strExpression) {
		// tokenize the string based on separator
		String[] tokens = strExpression.split(separator,4);
		// must be a triple
		if(tokens.length>=3 && tokens.length<=4) {
			TYPE type = TYPE.valueOf(tokens[0]);
			if(type!=null) {
				ValidatorExpression expression = null;
				if(type==TYPE.NS_PREFIX) {
					expression = new ValidatorExpression(type,tokens[1],tokens[2]);
				}
				else {
					LOG_LEVEL level = LOG_LEVEL.valueOf(tokens[1]);
					if(level!=null) {
						expression = new ValidatorExpression(type,level,tokens[2]);
					}
				}
				if(tokens.length==4) {
					expression.setDescription(tokens[3]);
				}
				if(expression!=null) {
					addExpression(expression);
				}
			}
		}
		else {
			logger.error("Invalid #tokens ("+tokens.length+") in expression "+strExpression);
		}
	}
	
	/**
	 * Add specified expression to collection 
	 * 
	 * @param expression
	 */
	public void addExpression(ValidatorExpression expression) {
		this.expressions.add(expression);
	}
	
	
	/**
	 * Performs the validation and logs results
	 * 
	 * @return
	 */
	public int validate() {
		int nErrors=0;
		// create namespace definition to be included
		// in each xpath/xquery
		String nsPrefixes = "";
		for(ValidatorExpression expression : expressions) {
			if(expression.getType()==TYPE.NS_PREFIX) {
				nsPrefixes += "declare namespace "+expression.getPrefix()+"='"+expression.getValue()+"';";
			}
		}
		// make sure XmlObjec is set
		if(this.xmlObject!=null) {
			
			// loop over expressions
			for(ValidatorExpression expression : expressions) {
				XmlCursor cursor = xmlObject.newCursor();
				
				String message=null;
				boolean success = false;

				switch(expression.getType()) {
				case XPATH:
					String xpath = nsPrefixes+expression.getValue();
					
					message = expression.getDescription();
					if(message!=null) message += ". XPATH:"+expression.getValue();
					else message =  expression.getValue();
					
					try {
						cursor.selectPath(xpath);
						if(cursor.getSelectionCount()==1) {
							cursor.toNextSelection();
							success = new Boolean(cursor.getTextValue());
						}
						else {
							message="XPath returned nothing or more than one object.";
						}
					}
					catch(RuntimeException e) {
						logger.error(e.getMessage());
					}
					break;
				case XQUERY:
					String xquery = nsPrefixes+expression.getValue();
				
					message = expression.getDescription();
					if(message!=null) message += ". XQUERY: "+expression.getValue();
					else message =  expression.getValue();
					
					try {
						XmlCursor result = cursor.execQuery(xquery);
						if(result!=null) {
							//logger.debug(result.isText());
							//logger.debug(result.getTextValue());
							success = new Boolean(result.getTextValue());
						}
					}
					catch(XmlRuntimeException e) {
						logger.error(e.getMessage());
					}
					break;
				case NS_PREFIX:
					break;
				}
				// log result
				if(message!=null) {
					if(success) {
						logger.info("PASSED: "+message);
					}
					else {
						message = "FAILED: "+message;
						switch(expression.getLevel()) {
						case ERROR:
							logger.error(message);
							break;
						case WARNING:
							logger.warn(message);
							break;
						case INFO:
							logger.info(message);
							break;
						}
					}
				}
				cursor.dispose();
			}
		}
		else {
			nErrors++;
			logger.error("XmlObject not set");
		}
		return nErrors;
	}
	
	
	//===============================================================
	// GETTERS/SETTERS
	//===============================================================
	public XmlObject getXmlObject() {
		return xmlObject;
	}
	public void setXmlObject(XmlObject xmlObject) {
		this.xmlObject = xmlObject;
	}
	public ArrayList<ValidatorExpression> getExpressions() {
		return expressions;
	}
	
	public String getSeparator() {
		return separator;
	}
	public void setSeparator(String separator) {
		this.separator = separator;
	}
}
