package org.openmetadata.util.xmlbeans;

import java.util.ArrayList;
import java.util.List;

import javax.xml.namespace.QName;

import org.apache.commons.lang.StringEscapeUtils;
import org.apache.xmlbeans.XmlBeans;
import org.apache.xmlbeans.XmlCursor;
import org.apache.xmlbeans.XmlCursor.TokenType;
import org.apache.xmlbeans.XmlException;
import org.apache.xmlbeans.XmlObject;
import org.apache.xmlbeans.XmlOptions;

public class XhtmlUtilities {

	/**
	 * Sets XHTML content (or a simple string) to a mixed content xml element.
	 * 
	 * @param xo
	 *            the xml object for the element which the mixed XHTML content
	 *            is to be set
	 * @param val
	 *            the value to set to the mixed content xml object
	 */
	public static void setXhtmlContent(XmlObject xo, String val) {
		XmlObject temp;
		try {
			temp = XmlObject.Factory.parse("<xml-fragment>" + val
					+ "</xml-fragment>");
			namespaceXHTMLContent(temp);
			XmlCursor cursorXo = xo.newCursor();
			cursorXo.removeXmlContents();
			cursorXo.toFirstContentToken();
			temp.newCursor().copyXmlContents(cursorXo);
			cursorXo.dispose();
		} catch (XmlException e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * Creates a new mixed content element with a given value.
	 * 
	 * @param xc
	 *            the type to be created
	 * @param value
	 *            the mixed content value to set to the newly created object
	 * @return a new object of the given type with the given content
	 */
	public static <T extends XmlObject> T createMixedContent(Class<T> xc,
			String value) {
		try {
			T xo = xc.cast(XmlBeans.getContextTypeLoader().parse(
					"<xml-fragment>" + value + "</xml-fragment>",
					XmlBeans.typeForClass(xc),
					new XmlOptions().setLoadReplaceDocumentElement(null)));
			namespaceXHTMLContent(xo);
			return xo;
		} catch (XmlException e) {
			throw new RuntimeException(e);
		}

	}

	/**
	 * Gets mixed XHTML content, stripping off XHTML namespaces.
	 * 
	 * @param xo
	 * @return the content as a string, with XHTML namespace prefixes and
	 *         declarations removed
	 */
	public static String getMixedContent(XmlObject xo) {
		String text = "";
		XmlCursor cursor = xo.copy().newCursor();
		cursor.toFirstAttribute();
		List<String> tags = new ArrayList<String>();
		while (cursor.toNextToken() != TokenType.ENDDOC) {
			TokenType type = cursor.currentTokenType();
			if (type.equals(TokenType.TEXT)) {
				text += StringEscapeUtils.escapeXml(cursor.getTextValue());
			} else if (type.equals(TokenType.START)) {
				String name = cursor.getName().getLocalPart();
				tags.add(name);
				text += "<" + name;
				if (cursor.toFirstAttribute()) {
					text += " " + cursor.getName().getLocalPart() + "=\""
							+ cursor.getTextValue() + "\"";
					while (cursor.toNextAttribute()) {
						text += " " + cursor.getName().getLocalPart() + "=\""
								+ cursor.getTextValue() + "\"";
					}
				}
				TokenType subType = cursor.toNextToken();
				if (subType.equals(TokenType.END)) {
					text += "/>";
					tags.remove(tags.size() - 1);
				} else {
					text += ">";
					cursor.toPrevToken();
				}
			} else if (type.equals(TokenType.END)) {
				if (tags.size() > 0) {
					text += "</" + tags.get(tags.size() - 1) + ">";
					tags.remove(tags.size() - 1);
				}
			}
		}
		cursor.dispose();
		return text;
	}

	/**
	 * Sets namespace information to a mixed XHTML content that does not
	 * currently have proper XHTML namespace information.
	 * 
	 * @param xo
	 *            the xml element which is to be namespaced
	 */
	public static void namespaceXHTMLContent(XmlObject xo) {
		XmlCursor cursor = xo.newCursor();
		cursor.toFirstAttribute();
		while (cursor.toNextToken() != TokenType.ENDDOC) {
			TokenType type = cursor.currentTokenType();
			if (type.equals(TokenType.START)) {
				String name = cursor.getName().getLocalPart();
				cursor.setName(new QName("http://www.w3.org/1999/xhtml", name,
						"xhtml"));
			} else if (type.equals(TokenType.END)) {
			}
		}
		cursor.dispose();
	}
}
