package org.openmetadata.beans.ddi.lifecycle.adt.impl;

import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

import org.apache.log4j.Logger;
import org.openmetadata.beans.ddi.lifecycle.factory.DdiBeanFactory;
import org.openmetadata.beans.ddi.lifecycle.reusable.IdentifiableBean;
import org.openmetadata.beans.exceptions.ResolverException;
import org.openmetadata.beans.impl.MutableBeanInitializer;
import org.openmetadata.beans.notification.ChangeEvent.Type;
import org.openmetadata.beans.notification.ChangeListener;
import org.openmetadata.beans.notification.impl.IdentifiableChangeEventImpl;

public class ResolvingSchemeItemListImpl<B extends IdentifiableBean> extends SchemeItemListImpl<B> {

	private static final Logger logger = Logger.getLogger(ResolvingSchemeItemListImpl.class);
	
	private Set<String> urnSet = new LinkedHashSet<String>();

	public ResolvingSchemeItemListImpl(String maintainableUrn, Class<? extends B> beanClass,
			MutableBeanInitializer beanInitializer,
			DdiBeanFactory beanFactory, ChangeListener listener) {
		super(maintainableUrn, beanClass, beanInitializer, beanFactory, listener);
	}
	
	@Override
	public void initAddIdentifiableUrn(String urn) {
		urnSet.add(urn);
	}

	@Override
	public void initSetIdentifiableUrns(String[] urns) {
		for (String urn : urns) {
			urnSet.add(urn);
		}
	}
	
	@Override
	public void initAddIdentifiableBean(B bean) {
		urnSet.add(bean.getUrn());
	};
	
	@Override
	public void initSetIdentifiableBeans(B[] beans) {
		for(B bean : beans){
			urnSet.add(bean.getUrn());
		}
	}

	@Override
	protected boolean add(B bean) {
		return urnSet.add(bean.getUrn());		
	}

	@Override
	public Iterator<B> iterator() {
		return new DynamicIdentifiableIterator<B>(urnSet, this.getBeanClass(),
				this.getResolver());
	}
	
	@Override
	public boolean contains(B bean) {
		return urnSet.contains(bean.getUrn());
	}

	@Override
	public boolean remove(B bean) {
		boolean result = urnSet.remove(bean.getUrn());
		if(result){
			for (IdentifiableBean referrer : bean.getReferrers()) {
				referrer.removeReference(bean);
			}
		}		
		this.notifyChange(new IdentifiableChangeEventImpl(Type.DELETE, bean));
		return result;
	}

	@Override
	public void moveBefore(B move, B before) {
		Set<String> tmpSet = new LinkedHashSet<String>();
		tmpSet.addAll(urnSet);
		
		urnSet.clear();
		for(String urn : tmpSet){
			if(urn.equals(move.getUrn()) == false){
				if(urn.equals(before.getUrn())){
					urnSet.add(move.getUrn());
				}
				urnSet.add(urn);
			}
		}
		this.ddiBeanChanged();
	}

	@Override
	public void moveAfter(B move, B after) {
		Set<String> tmpSet = new LinkedHashSet<String>();
		tmpSet.addAll(urnSet);
		
		urnSet.clear();
		for(String urn : tmpSet){
			if(urn.equals(move.getUrn()) == false){				
				urnSet.add(urn);
				if(urn.equals(after.getUrn())){
					urnSet.add(move.getUrn());
				}
			}
		}
		this.ddiBeanChanged();
	}

	@Override
	public int size() {
		return urnSet.size();
	}

	@Override
	public void sort(Comparator<B> comparator) {
			
		B[] tmpArr = this.toArray();		
		List<B> tmpList = Arrays.asList(tmpArr);
		Collections.sort(tmpList, comparator);
		// clear all the urns in the set
		urnSet.clear();
		for(B bean : tmpList){
			urnSet.add(bean.getUrn());
		}
		this.ddiBeanChanged();
	}

	private List<B> getTempList() throws ResolverException {
		List<B> list = new ArrayList<B>();
		for (String urn : urnSet) {
			B bean = this.getResolver().resolve(this.getBeanClass(), urn);
			list.add(bean);			
		}
		return list;
	}

	@Override
	public B[] toArray() {
		@SuppressWarnings("unchecked")
		B[] typeArr = (B[]) Array.newInstance(this.getBeanClass(),0);		
		try {
			return this.getTempList().toArray(typeArr);
		} catch (ResolverException e) {
			logger.error(e.getMessage(), e);
			throw new RuntimeException(e);
		}
	}

	@Override
	public String[] getUrns() {
		return urnSet.toArray(new String[0]);
	}

	

}