package org.openmetadata.beans.ddi.lifecycle.adt.impl;

import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;

import org.openmetadata.beans.ddi.lifecycle.DdiBean;
import org.openmetadata.beans.ddi.lifecycle.adt.DdiBeanList;
import org.openmetadata.beans.ddi.lifecycle.factory.DdiBeanFactory;
import org.openmetadata.beans.ddi.lifecycle.impl.DdiBeanImpl;
import org.openmetadata.beans.impl.MutableBeanInitializer;
import org.openmetadata.beans.notification.ChangeListener;

public abstract class DdiBeanListImpl<B extends DdiBean> extends DdiBeanImpl
	implements DdiBeanList<B> {
	
	private final List<B> list;
	private final Class<B> beanClass;

	protected abstract B createNew();

	public DdiBeanListImpl(Class<B> beanClass, 
			MutableBeanInitializer beanInitializer,
			DdiBeanFactory factory,
			ChangeListener beanChangeListener) {
		super(beanInitializer, factory, beanChangeListener);
		this.list = new ArrayList<B>();
		this.beanClass = beanClass;
	}	

	protected List<B> getInternalList() {
		return list;
	}

	@Override
	public final B addNew() {
		B bean = createNew();
		if (bean != null) {
			this.add(bean);
		}
		this.ddiBeanChanged();
		return bean;
	}

	protected boolean add(B bean) {
		boolean returnedVal = list.add(bean);
		if(returnedVal){
			this.ddiBeanChanged();
		}
		return returnedVal;
	}

	@Override
	public boolean contains(B bean) {
		return list.contains(bean);
	};

	@Override
	public Iterator<B> iterator() {
		return list.iterator();
	}

	@Override
	public boolean remove(B bean) {
		boolean isRemoved = list.remove(bean);
		if (isRemoved) {
			this.ddiBeanChanged();
		}
		return isRemoved;
	}

	@Override
	public void moveBefore(B move, B before) {
		if (move == before)
			return;

		if (list.contains(move) && list.contains(before)) {
			list.remove(move);
			int beforeIndex = list.lastIndexOf(before);
			list.add(beforeIndex, move);
			
			this.ddiBeanChanged();
		}
	}

	@Override
	public void moveAfter(B move, B after) {
		if (move == after)
			return;

		if (list.contains(move) && list.contains(after)) {
			list.remove(move);
			int afterIndex = list.lastIndexOf(after);
			if (afterIndex < list.size() - 1) {
				// indexes: 0 to S-1
				// when size()is S then afterIndex is at most S-1
				// if afterIndex is S-1(rightmost item) then it will insert at S
				list.add(afterIndex + 1, move);
			} else {
				list.add(move);
			}
			
			this.ddiBeanChanged();
		}
	}
	
	@Override
	public B[] toArray() {
		@SuppressWarnings("unchecked")
		B[] typeArr = (B[]) Array.newInstance(beanClass,0);		
		return list.toArray(typeArr);
	};

	@Override
	public int size() {
		return list.size();
	}

	@Override
	public void sort(Comparator<B> comparator) {
		Collections.sort(list, comparator);
		this.ddiBeanChanged();
	}
}
