package org.openmetadata.beans.ddi.lifecycle.reusable.impl;

import org.openmetadata.beans.ddi.lifecycle.factory.DdiBeanFactory;
import org.openmetadata.beans.ddi.lifecycle.notification.ReferenceAddedEvent;
import org.openmetadata.beans.ddi.lifecycle.notification.ReferenceRemovedEvent;
import org.openmetadata.beans.ddi.lifecycle.notification.ReferenceReplacedEvent;
import org.openmetadata.beans.ddi.lifecycle.reusable.IdentifiableBean;
import org.openmetadata.beans.ddi.lifecycle.reusable.ReferenceBean;
import org.openmetadata.ddi.util.exceptions.URNFormatException;
import org.openmetadata.ddi_3_1.util.URN;
import org.openmetadata.beans.ddi.lifecycle.utility.CompareUtil;
import org.apache.commons.lang3.StringUtils;
import org.apache.log4j.Logger;
import org.openmetadata.beans.exceptions.ResolverException;
import org.openmetadata.beans.notification.ChangeListener;

public class ReferenceBeanImpl<B extends IdentifiableBean> extends
		UnsettableDdiBeanImpl implements ReferenceBean<B> {
	private static final Logger logger = Logger.getLogger(ReferenceBeanImpl.class);

	private static final boolean DEFAULT_IS_EXTERNAL = false;
	private static final boolean DEFAULT_IS_REFERENCE = true;
	private static final boolean DEFAULT_LATE_BOUND = false;

	private Boolean isExternal;
	private Boolean isReference;
	private Boolean lateBound;
	private String objectLanguage;
	private String sourceContext;

	private ReferenceBean<IdentifiableBean> scheme;

	private Class<B> beanClass;

	private URN urn;

	public ReferenceBeanImpl(Class<B> beanClass, DdiBeanFactory factory,
			ChangeListener listener) {
		super(factory, listener);
		this.beanClass = beanClass;
	}

	public void setReferenceUrn(String newUrnStr) throws URNFormatException {
		String toRemoveUrn = (urn != null) ? urn.toString() : null;
		urn = URN.getURN(newUrnStr);
		this.notifyChange(new ReferenceReplacedEvent(newUrnStr, toRemoveUrn));
	}

	@Override
	public boolean isReferring(B bean) {
		return (urn != null && bean != null) ? urn.toString().equalsIgnoreCase(
				bean.getUrn()) : false;
	}

	@Override
	public boolean getIsExternal() {
		if (isSetIsExternal()) {
			return isExternal;
		} else {
			return DEFAULT_IS_EXTERNAL;
		}
	}

	@Override
	public void setIsExternal(boolean isExternal) {
		if (CompareUtil.areDifferentValues(this.isExternal, isExternal)) {
			this.isExternal = isExternal;
			populate();
			this.ddiBeanChanged();
		}
	}

	@Override
	public boolean isSetIsExternal() {
		return isExternal != null;
	}

	@Override
	public boolean getIsReference() {
		if (isSetIsReference()) {
			return isReference;
		} else {
			return DEFAULT_IS_REFERENCE;
		}
	}

	@Override
	public void setIsReference(boolean isReference) {
		if (CompareUtil.areDifferentValues(this.isReference, isReference)) {
			this.isReference = isReference;
			populate();
			this.ddiBeanChanged();
		}
	}

	@Override
	public boolean isSetIsReference() {
		return isReference != null;
	}

	@Override
	public boolean getLateBound() {
		if (isSetLateBound()) {
			return lateBound;
		} else {
			return DEFAULT_LATE_BOUND;
		}
	}

	@Override
	public void setLateBound(boolean boo) {
		if (CompareUtil.areDifferentValues(this.lateBound, boo)) {
			this.lateBound = boo;
			populate();
			this.ddiBeanChanged();
		}
	}

	@Override
	public boolean isSetLateBound() {
		return lateBound != null;
	}

	@Override
	public String getObjectLanguage() {
		return StringUtils.defaultString(objectLanguage);
	}

	@Override
	public void setObjectLanguage(String value) {
		if (CompareUtil.areDifferentValues(this.objectLanguage, value)) {
			this.objectLanguage = value;
			populate();
			this.ddiBeanChanged();
		}
	}

	@Override
	public boolean isSetObjectLanguage() {
		return objectLanguage != null;
	}

	@Override
	public String getSourceContext() {
		return sourceContext;
	}

	@Override
	public void setSourceContext(String value) {
		if (CompareUtil.areDifferentValues(this.sourceContext, value)) {
			this.sourceContext = value;
			populate();
			this.ddiBeanChanged();
		}
	}

	@Override
	public boolean isSetSourceContext() {
		return sourceContext != null;
	}

	@Override
	public boolean isSetScheme() {
		return scheme != null;
	}

	@Deprecated
	@Override
	public ReferenceBean<IdentifiableBean> getScheme() {
		return scheme;
	}

	@Override
	public boolean isSetUrn() {
		return urn != null;
	}

	@Override
	public String getUrn() {
		return (urn != null) ? urn.toString() : "";
	}

	private void internalSetReferenceTo(B bean) throws URNFormatException {
		assert (bean != null);

		if (urn != null) {
			if (urn.toString().equals(bean.getUrn()) == false) {
				String toRemoveUrn = urn.toString();
				urn = URN.getURN(bean.getUrn());
				this.notifyChange(new ReferenceReplacedEvent(bean.getUrn(),
						toRemoveUrn));
				this.populate();
			} else {
				// do nothing since the urns are the same
			}
		} else {
			urn = URN.getURN(bean.getUrn());
			this.notifyChange(new ReferenceAddedEvent(bean.getUrn()));
			this.populate();
		}
	}

	@Override
	public void setReferenceTo(B bean)  {
		if (bean != null) {
			try {
				this.internalSetReferenceTo(bean);
			} catch (URNFormatException e) {
				logger.error(e.getMessage(), e);
				throw new RuntimeException(e);
			}
		} else {
			logger.error("The given bean is null.");
			throw new RuntimeException("Bean must not be null.");
		}
	}

	@Override
	public B getReferredObject() throws ResolverException {
		if (urn != null) {
			String urn = this.getUrn();
			B idBean = this.getResolver().resolve(beanClass, urn);
			if (idBean != null) {
				return idBean;
			}
		}
		throw new ResolverException();
	}
	
	@Override
	protected void doInternalUnset() {
		super.doInternalUnset();
		if (urn != null) {
			this.notifyChange(new ReferenceRemovedEvent(urn.toString()));
			urn = null;
		}
	}

	@Override
	final public boolean internalIsSet() {
		return urn != null;
	}
}
