package org.openmetadata.beans.ddi.lifecycle.reusable.impl;

import java.util.HashSet;
import java.util.Set;

import org.openmetadata.beans.ddi.lifecycle.adt.ResolvingSet;
import org.openmetadata.beans.ddi.lifecycle.factory.DdiBeanFactory;
import org.openmetadata.beans.ddi.lifecycle.notification.ReferenceChangeEvent;
import org.openmetadata.beans.ddi.lifecycle.reusable.ActionCode;
import org.openmetadata.beans.ddi.lifecycle.reusable.IdentifiableBean;
import org.openmetadata.beans.ddi.lifecycle.reusable.LabelBean;
import org.openmetadata.beans.ddi.lifecycle.reusable.NameBean;
import org.openmetadata.beans.ddi.lifecycle.reusable.StructuredStringBean;
import org.openmetadata.ddi_3_1.util.URN;
import org.openmetadata.beans.impl.MutableBeanInitializer;
import org.openmetadata.beans.impl.MutableIdentifiableBeanImpl;
import org.openmetadata.beans.notification.ChangeEvent;
import org.openmetadata.beans.notification.IdentifiableChangeEvent;
import org.openmetadata.beans.reference.Resolver;

public abstract class AbstractIdentifiableBeanImpl extends
		MutableIdentifiableBeanImpl implements IdentifiableBean {

	private final String id;
	private final String urn;

	private String objectSource;
	private ActionCode action;
	
	private AttributeSetImpl attrSet;

	private NameValueBeanImpl name;
	private LabelValueBeanImpl label;
	private StructuredStringValueBeanImpl description;

	private UserIDValueBeanImpl userIdValues;

	private final DdiBeanFactory beanFactory;
	private final MutableBeanInitializer beanInitializer;

	protected abstract void doChangeReference(IdentifiableBean toRemove,
			IdentifiableBean toAdd) ;

	protected abstract void doRemoveReference(IdentifiableBean toRemove);

	private static URN getParentURN(URN urn) {
		if (urn.hasParent()) {
			return urn.getParent();
		} else {
			return urn;
		}
	}

	public AbstractIdentifiableBeanImpl(Boolean newInstance, URN urn,
			MutableBeanInitializer beanInitializer,
			DdiBeanFactory beanFactory) {
		super(newInstance, urn.toString(), getParentURN(urn).toString(),
				beanInitializer);
		this.beanInitializer = beanInitializer;
		this.beanFactory = beanFactory;

		this.id = urn.getId();
		this.urn = urn.toString();
		
		attrSet = new AttributeSetImpl();

		name = new NameValueBeanImpl(beanFactory, this);
		label = new LabelValueBeanImpl(beanFactory, this);
		description = new StructuredStringValueBeanImpl(beanFactory, this);

		userIdValues = new UserIDValueBeanImpl(beanFactory, this);
	}
	
	protected MutableBeanInitializer getBeanInitializer(){
		return beanInitializer;
	}

	protected DdiBeanFactory getBeanFactory() {
		return beanFactory;
	}

	protected Resolver getResolver() {
		return beanFactory;
	}

	@Override
	protected final void handleChange(IdentifiableChangeEvent event) {
		// TODO ? .... 
		// This method might be very confusing for the users
	}
	
	@Override
	public AttributeSetImpl getAttributeSet() {
		return this.attrSet;
	}

	@Override
	public IdentifiableBean[] getReferrers() {
		Set<String> referringUrnSet = beanFactory.getReferrers(this.getUrn());
		Set<IdentifiableBean> set;
		try {
			set = beanFactory.resolve(IdentifiableBean.class, referringUrnSet);
			return set.toArray(new IdentifiableBean[0]);
		} catch (org.openmetadata.beans.exceptions.ResolverException e1) {
			return new IdentifiableBean[0];
		}
	}
	
	@Override
	public ResolvingSet<IdentifiableBean> getReferrerSet() {
		Set<String> referringUrnSet = beanFactory.getReferrers(this.getUrn());
		return new ResolvingSet<IdentifiableBean>(
				referringUrnSet, IdentifiableBean.class, beanFactory);
	}

	@Override
	public final String getId() {
		return id;
	}

	@Override
	public final String getUrn() {
		return urn;
	}

	@Override
	public boolean isSetUrn() {
		return urn != null;
	}

	@Override
	public String getObjectSource() {
		return (objectSource != null) ? objectSource : "";
	}

	@Override
	public void setObjectSource(String value) {
		this.objectSource = value;
		this.change();
	}

	@Override
	public boolean isSetObjectSource() {
		return objectSource != null;
	}

	@Override
	public ActionCode getAction() {
		return action;
	}

	@Override
	public boolean isSetAction() {
		return action != null;
	}

	@Override
	public void setAction(ActionCode ac) {
		if (action != ac) {
			action = ac;
			this.change();
		}
	}

	@Override
	public UserIDValueBeanImpl getUserIDValues() {
		return this.userIdValues;
	}

	@Override
	public NameValueBeanImpl getName() {
		return name;
	}

	@Override
	public LabelValueBeanImpl getLabel() {
		return label;
	}

	@Override
	public StructuredStringValueBeanImpl getDescription() {
		return description;
	}

	@Override
	public void copyNameLabelDescription(IdentifiableBean from) {
		this.getName().unset();
		for (NameBean name : from.getName().getAllValues()) {
			this.getName().getBean(name.getLanguage())
					.setStringValue(name.getStringValue());
		}
		this.getLabel().unset();
		for (LabelBean label : from.getLabel().getAllValues()) {
			this.getLabel().getBean(label.getLanguage())
					.setXhtmlContent(label.getXhtmlContent());
		}
		this.getDescription().unset();
		for (StructuredStringBean sstr : from.getDescription().getAllValues()) {
			this.getDescription().getBean(sstr.getLanguage())
					.setXhtmlContent(sstr.getXhtmlContent());
		}
	}

	@Override
	public final void changeReference(IdentifiableBean current,
			IdentifiableBean another)  {
		if (current != null) {
			if (another != null) {
				this.doChangeReference(current, another);
			} else {
				this.doRemoveReference(current);
			}
		}
	}

	@Override
	public final void removeReference(IdentifiableBean toRemove) {
		if (toRemove != null) {
			this.doRemoveReference(toRemove);
		}
	}

	@Override
	protected Set<String> getReferenceAdditions(ChangeEvent event) {
		Set<String> additions = new HashSet<String>();
		if ((event instanceof IdentifiableChangeEvent) == false) {
			if (event instanceof ReferenceChangeEvent) {
				ReferenceChangeEvent refChangeEvent = (ReferenceChangeEvent) event;
				additions.addAll(refChangeEvent.getAddedReferences());
			}
			for (ChangeEvent initiatingEvent : event.getInitiatingEvents()) {
				additions.addAll(this.getReferenceAdditions(initiatingEvent));
			}
		}
		return additions;
	}

	@Override
	protected Set<String> getReferenceDeletions(ChangeEvent event) {
		Set<String> deletions = new HashSet<String>();
		if ((event instanceof IdentifiableChangeEvent) == false) {
			if (event instanceof ReferenceChangeEvent) {
				ReferenceChangeEvent refChangeEvent = (ReferenceChangeEvent) event;
				deletions.addAll(refChangeEvent.getRemovedReferences());
			}
			for (ChangeEvent initiatingEvent : event.getInitiatingEvents()) {
				deletions.addAll(this.getReferenceDeletions(initiatingEvent));
			}
		}
		return deletions;
	}
	
}
