package org.openmetadata.beans.ddi.lifecycle.adt.impl;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

import org.apache.log4j.Logger;
import org.openmetadata.beans.ddi.lifecycle.adt.ReferenceSet;
import org.openmetadata.beans.ddi.lifecycle.factory.DdiBeanFactory;
import org.openmetadata.beans.ddi.lifecycle.impl.DdiBeanImpl;
import org.openmetadata.beans.ddi.lifecycle.notification.ReferenceAddedEvent;
import org.openmetadata.beans.ddi.lifecycle.notification.ReferenceRemovedEvent;
import org.openmetadata.beans.ddi.lifecycle.reusable.IdentifiableBean;
import org.openmetadata.beans.exceptions.ResolverException;
import org.openmetadata.beans.impl.MutableBeanInitializer;
import org.openmetadata.beans.notification.ChangeListener;

public class ReferenceSetImpl<B extends IdentifiableBean> extends DdiBeanImpl
		implements ReferenceSet<B> {

	private static Logger logger = Logger.getLogger(ReferenceSetImpl.class);
	private Set<String> urnSet = new LinkedHashSet<String>();
	private Class<B> beanClass;

	public ReferenceSetImpl(Class<B> beanClass, 
			MutableBeanInitializer beanInitializer,
			DdiBeanFactory factory,			
			ChangeListener listener) {
		super(beanInitializer, factory, listener);
		this.beanClass = beanClass;			
	}

	public void initReferenceUrns(String[] urns) {
		for (String urn : urns) {
			urnSet.add(urn);
		}		
	}

	@Override
	public String[] getUrns() {
		return urnSet.toArray(new String[0]);
	}

	@Override
	public boolean contains(B bean) {
		return (bean != null) ? urnSet.contains(bean.getUrn()) : false;
	};

	@Override
	public boolean contains(String urn) {
		return urnSet.contains(urn);
	}

	@Override
	public boolean add(B bean) {
		Set<String> addedSet = new HashSet<String>();
		addedSet.add(bean.getUrn());
		boolean result = urnSet.add(bean.getUrn());
		this.notifyChange(new ReferenceAddedEvent(bean.getUrn()));
		return result;
	}

	@Override
	public boolean remove(B bean) {
		Set<String> toDeleteSet = new HashSet<String>();
		toDeleteSet.add(bean.getUrn());
		boolean result = urnSet.remove(bean.getUrn());
		this.notifyChange(new ReferenceRemovedEvent(bean.getUrn()));
		return result;
	}

	@Override
	public boolean remove(String urn) {
		Set<String> toDeleteSet = new HashSet<String>();
		toDeleteSet.add(urn);
		boolean result = urnSet.remove(urn);
		this.notifyChange(new ReferenceRemovedEvent(urn));
		return result;
	}

	@Override
	public B[] toArray(B[] arr) {
		List<B> list = new ArrayList<B>();
		for (String urn : urnSet) {
			try {
				B bean = this.getResolver().resolve(beanClass, urn);
				list.add(bean);
			} catch (ResolverException e) {
				logger.error(e.getMessage(), e);
				throw new RuntimeException(e);
			}
		}
		return list.toArray(arr);
	};

	@Override
	public Iterator<B> iterator() {
		return new DynamicIdentifiableIterator<B>(urnSet, beanClass,
				this.getResolver());
	}

	@Override
	public int size() {
		return urnSet.size();
	}

	@Override
	public void clear() {
		Set<String> deletedSet = new HashSet<String>();
		deletedSet.addAll(urnSet);
		urnSet.clear();
		this.notifyChange(new ReferenceRemovedEvent(deletedSet));
	}
	
	@Override
	public void moveBefore(B move, B before) {
		if(move.equals(before)){
			return;
		}else if(move.getUrn().equals(before.getUrn())){
			return;
		}
		
		Set<String> tmpSet = new LinkedHashSet<String>();
		tmpSet.addAll(urnSet);
		
		urnSet.clear();
		for(String urn : tmpSet){
			if(urn.equals(move.getUrn()) == false){
				if(urn.equals(before.getUrn())){
					urnSet.add(move.getUrn());
				}
				urnSet.add(urn);
			}
		}
		this.ddiBeanChanged();
	}

	@Override
	public void moveAfter(B move, B after) {
		if(move.equals(after)){
			return;
		}else if(move.getUrn().equals(after.getUrn())){
			return;
		}
		
		Set<String> tmpSet = new LinkedHashSet<String>();
		tmpSet.addAll(urnSet);
		
		urnSet.clear();
		for(String urn : tmpSet){
			if(urn.equals(move.getUrn()) == false){				
				urnSet.add(urn);
				if(urn.equals(after.getUrn())){
					urnSet.add(move.getUrn());
				}
			}
		}
		this.ddiBeanChanged();
	}
}