package org.openmetadata.util.xmlbeans;

import org.apache.xmlbeans.XmlCursor;
import org.apache.xmlbeans.XmlObject;
import org.apache.xmlbeans.XmlCursor.TokenType;

public class XmlCleaner {

	/**
	 * Cleans unnecessary whitespace from elements using the
	 * {@link String#trim()} method. This is intended to be used on XML
	 * documents that are formatted for "readability" and contain whitespace in
	 * elements that is potentially problematic.
	 * <p>
	 * <b>Example:</b><br/>
	 * <br/>
	 * &lt;id><br/>
	 * &nbsp;&nbsp;&nbsp;TEST<br/>
	 * &lt;/id><br/>
	 * <br/>
	 * would be cleaned to<br/>
	 * <br/>
	 * &lt;id>TEST&lt;/id>
	 * </p>
	 * 
	 * @param xmlObject
	 *            the XML instance to be cleaned; this is copied so that the
	 *            original is left unmodified
	 * @param copy
	 *            <code>true</code> if the instance should be copied before
	 *            cleaning, leaving the original unmodified; <code>false</code>
	 *            to change the original instance
	 * @return the cleaned instance
	 */
	public static XmlObject cleanWhitespace(XmlObject xmlObject, boolean copy) {
		if (copy) {
			xmlObject = xmlObject.copy();
		}
		XmlCursor cursor = xmlObject.newCursor();
		TokenType token;
		try {
			cursor.toStartDoc();
			while ((token = cursor.toNextToken()) != TokenType.NONE) {
				if (token == TokenType.TEXT) {
					String cleanedValue = cursor.getTextValue().trim();
					cursor.removeChars(cursor.getTextValue().length());
					cursor.insertChars(cleanedValue);
					cursor.toPrevToken();
				}
			}
			cursor.toStartDoc();
			return cursor.getObject();
		} catch (Exception e) {
			throw new RuntimeException("Error cleaning the XML", e);
		} finally {
			cursor.dispose();
		}

	}

}
