package org.openmetadata.store.repository;

import java.util.Set;

import org.openmetadata.store.exceptions.ObjectNotFoundException;
import org.openmetadata.store.exceptions.StoreException;
import org.openmetadata.store.snapshot.SnapshotHistory;

/**
 * @author Jack Gager
 * 
 * @param <Item>
 */
public interface SnapshotRepository<Item extends Object> extends
		MutableRepository<Item> {

	/**
	 * Performs the same function of the {@link #save(String, Set, Set, Set)}
	 * method, but will also preserve a snapshot of the objects saved by the
	 * invocation. This snapshot will be part of the {@link SnapshotHistory} for
	 * this repository.
	 * 
	 * @param contextId
	 *            the identifier of the context object in which the save is
	 *            being made; note that it is possible that a child context
	 *            object may be being saved- this parameter is always the
	 *            highest level context identifier
	 * @param additions
	 *            the set of <code>Item</code>s which are to be newly added to
	 *            the repository
	 * @param updates
	 *            the set of <code>Items</code>s which are to be updated in the
	 *            repository
	 * @param deletions
	 *            the set of <code>Items</code> which are to be deleted from the
	 *            repository
	 * @param comment
	 *            used to capture a short comment about or a description of the
	 *            changes being saved
	 * @return a unique identifier for the save operation; this will be used to
	 *         relate raised events to a specific call to this method
	 * @throws StoreException
	 *             if the save operation fails due to issues with the
	 *             <code>Item</code>s to be saved
	 */
	public String save(String contextId, Set<Item> additions,
			Set<Item> updates, Set<Item> deletions, String comment)
			throws StoreException;

	/**
	 * Retrieves a snapshot version of an object from the repository.
	 * 
	 * @param snapshotId
	 *            the identifier of the snapshot from which the object is to be
	 *            retrieved
	 * @param id
	 *            the unique identifier of the object
	 * @return the object as a <code>Item</code> object
	 * @throws ObjectNotFoundException
	 *             if an object with the provided identifier does not exist in
	 *             the repository
	 */
	public Item getSnapshot(String snapshotId, String id)
			throws ObjectNotFoundException;

	/**
	 * Restores the data from a snapshot to the specified items, or if no item
	 * identifiers are provided restores all changes from the identified
	 * snapshot.
	 * <p>
	 * After calling this method, calls to the {@link #get(Set)} and
	 * {@link #get(String)} methods must represent the information that was
	 * saved at the point of the snapshot.
	 * 
	 * @param snapshotId
	 *            the identifier of the snapshot from which the identified beans
	 *            are to be restored
	 * @param itemIds
	 *            identifiers of the items from the snapshot which are to be
	 *            restored; if this is empty, all items from the snapshot will
	 *            be restored
	 */
	public void restoreSnapshot(String snapshotId, String... itemIds);

	/**
	 * Retrieves snapshot information available for the repository.
	 * 
	 * @return the available snapshots which can be restored
	 */
	public SnapshotHistory getSnapshotHistory();

}
