package org.openmetadata.store.repository;

import java.util.Set;

import org.openmetadata.store.catalog.CatalogProvider;
import org.openmetadata.store.exceptions.InvalidCriteriaException;
import org.openmetadata.store.exceptions.ObjectNotFoundException;
import org.openmetadata.store.query.Criteria;
import org.openmetadata.store.query.SearchResult;

/**
 * This is a repository for any identifiable objects. The objects in this
 * repository must have a unique identifier from which they can be retrieved.
 * 
 * @author Jack Gager
 * 
 * @param <Item>
 *            the base object for which any object which exists in the
 *            repository derives from.
 */
public interface Repository<Item extends Object> extends CatalogProvider {
	
	public boolean contains(String id);
	
	/**
	 * Retrieves an object from the repository by its unique identifier.
	 * 
	 * @param id
	 *            the unique identifier of the object
	 * @return the object as a <code>Item</code> object
	 * @throws ObjectNotFoundException
	 *             if an object with the provided identifier does not exist in
	 *             the repository
	 */
	public Item get(String id) throws ObjectNotFoundException;

	/**
	 * Retrieves a set of objects from the repository by their unique
	 * identifiers.
	 * 
	 * @param ids
	 *            the set of identifiers for the objects
	 * @return the objects as a set of <code>Item</code> objects
	 * @throws ObjectNotFoundException
	 *             if any of the requests objects cannot be found in the
	 *             repository; this must contain the identifiers of all objects
	 *             which cannot be found from the supplied set
	 */
	public Set<Item> get(Set<String> ids) throws ObjectNotFoundException;

	/**
	 * Queries for a the set of item identifiers which match the provided
	 * criteria.
	 * 
	 * @param contextId
	 *            the identifier of the context object in which the search is to
	 *            be performed
	 * @param criteria
	 *            the criteria an item must match to be included in the result
	 *            set
	 * @return an object containing the set of item identifiers and a boolean
	 *         indicating whether it is complete, which can be used to retrieve
	 *         the items from this repository from the {@link #get(String)} or
	 *         {@link #get(Set)} method.
	 * @throws InvalidCriteriaException
	 *             if the supplied criteria cannot be evaluated.
	 */
	public SearchResult query(String contextId, Criteria<?> criteria)
			throws InvalidCriteriaException;

}
