package org.openmetadata.store.repository;

import java.util.Set;

import org.openmetadata.store.exceptions.StoreException;
import org.openmetadata.store.repository.notification.SaveEvent;
import org.openmetadata.store.repository.notification.SaveNotifier;

/**
 * This is a repository for any identifiable objects which allows updates. The
 * objects in this repository must have a unique identifier from which they can
 * be retrieved.
 * <p>
 * This repository will notify all registered listeners of a successful save
 * operation via a {@link SaveEvent}. A single event must be raised at the
 * completion of each save and it must contain the unique identifiers of all
 * objects saved during that operation.
 * <p>
 * Before raising the <code>SaveEvent</code> the save method must return the
 * unique identifier for the specific save operation being performed. This
 * identification must be returned from the
 * {@link SaveEvent#getEventIdentifier()} method. This allows listeners of this
 * repository to ignore <code>SaveEvent</code>s raised due to their own calls to
 * the save method.
 * 
 * @author Jack Gager
 * 
 * @param <Item>
 *            the base object for which any object which exists in the
 *            repository derives from.
 */
public interface MutableRepository<Item extends Object> extends
		Repository<Item>, SaveNotifier {
	
	public String getBaseContext();
	
	/**
	 * Saves the supplied <code>Item</code> objects to the repository and
	 * returns a unique identifier for the operation. If any of the requested
	 * changes cannot be saved, all supplied changes to this method must be
	 * discarded.
	 * <p>
	 * Implementations must return the unique identifier for the event prior to
	 * raising any {@link SaveEvent} notifications.
	 * 
	 * @param contextId
	 *            the identifier of the context object in which the save is
	 *            being made; note that it is possible that a child context
	 *            object may be being saved- this parameter is always the
	 *            highest level context identifier
	 * @param additions
	 *            the set of <code>Item</code>s which are to be newly added to
	 *            the repository
	 * @param updates
	 *            the set of <code>Items</code>s which are to be updated in the
	 *            repository
	 * @param deletions
	 *            the set of <code>Items</code> which are to be deleted from the
	 *            repository
	 * @return a unique identifier for the save operation; this will be used to
	 *         relate raised events to a specific call to this method
	 * @throws StoreException
	 *             if the save operation fails due to issues with the
	 *             <code>Item</code>s to be saved
	 */
	public String save(String contextId, Set<Item> additions,
			Set<Item> updates, Set<Item> deletions) throws StoreException;

}
