package org.openmetadata.store.query.impl;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.openmetadata.beans.IdentifiableBean;
import org.openmetadata.store.query.Criteria;
import org.openmetadata.store.query.PropertySearch;
import org.openmetadata.store.query.ReferenceSet;
import org.openmetadata.store.query.Sort;
import org.openmetadata.store.query.TextSearch;

public class CriteriaImpl<B extends IdentifiableBean> implements Criteria<B> {

	private final Class<B> beanClass;
	private final HashSet<Class<? extends B>> beanClassSet;
	private final int pageSize;
	private final int maxCount;
	private final ArrayList<Sort> sortList;
	private final ReferenceSetImpl references;
	private final ReferenceSetImpl referrers;
	private final TextSearch text;
	private final HashSet<PropertySearch> properties;
	private boolean searchChildContext;
	private boolean searchParentContext;
	
	public CriteriaImpl(Class<B> beanClass) {
		this(beanClass, 0, 0, null);
	}

	public CriteriaImpl(Class<B> beanClass, int pageSize, int maxCount, TextSearch text) {
		this.beanClass = beanClass;
		this.pageSize = pageSize;
		this.maxCount = maxCount;
		this.sortList = new ArrayList<Sort>();
		this.references = new ReferenceSetImpl();
		this.referrers = new ReferenceSetImpl();
		this.properties = new HashSet<PropertySearch>();
		this.text = text;
		beanClassSet = new HashSet<Class<? extends B>>();
		beanClassSet.add(beanClass);
	}
	
	@Override
	public Class<B> getBeanClass() {
		return beanClass;
	}
	
	@Override
	public Set<Class<? extends B>> getBeanClasses() {
		return new HashSet<Class<? extends B>>(beanClassSet);
	}

	@Override
	public int getPageSize() {
		return pageSize;
	}

	@Override
	public int getMaxCount() {
		return maxCount;
	}
	
	@Override
	public List<Sort> getSort() {
		return new ArrayList<Sort>(sortList);
	}

	@Override
	public boolean searchParentContexts() {
		return searchParentContext;
	}

	@Override
	public boolean searchChildContexts() {
		return searchChildContext;
	}
	
	@Override
	public ReferenceSet getReferences() {
		return references;
	}

	@Override
	public ReferenceSet getReferrers() {
		return referrers;
	}

	@Override
	public boolean searchText() {
		return text!=null;
	}

	@Override
	public TextSearch getSearchText() {
		return text;
	}
	
	@Override
	public Set<PropertySearch> getSearchProperties() {
		return properties;
	}
	
	public void addBeanClass(Class<? extends B> beanClass) {
		beanClassSet.add(beanClass);
	}
	
	public void addSort(Sort sort) {
		sortList.add(sort);
	}
	
	public void setReferencesDirect(boolean directReferences) {
		this.references.setDirectReferences(directReferences);
	}
	
	public void addReference(Set<String> references) {
		this.references.addIds(references);
	}
	
	public void setReferrersDirect(boolean directReferrers) {
		this.referrers.setDirectReferences(directReferrers);
	}
	
	public void addReferrer(Set<String> referrers) {
		this.referrers.addIds(referrers);
	}
	
	public void addProperty(PropertySearch property) {
		properties.add(property);
	}
	
	public void setSearchParentContext(boolean searchParentContext) {
		this.searchParentContext = searchParentContext;
	}
	
	public void setSearchChildContext(boolean searchChildContext) {
		this.searchChildContext = searchChildContext;
	}
	
}
