package org.openmetadata.store.query;

import java.util.Set;

import org.openmetadata.store.exceptions.InvalidCriteriaException;

/**
 * Allows for full text searching of specified text fields within a queried
 * object. The enumerations for the language and fields must be agreed upon by
 * the provider and processor of the query. If an unrecognized enumeration is
 * used, then the processor must throw a {@link InvalidCriteriaException}.
 * <p>
 * Implementations must limit the search to the specified languages and fields.
 * If theses sets are empty, then it is assumed that the search should be
 * performed on all text fields for all languages.
 * <p>
 * The interpretation of the search text must also be agreed upon by the query
 * provider and processor.
 * 
 * @author Jack Gager
 * 
 */
public interface TextSearch {

	/**
	 * Retrieves the languages for which the text search should be performed. If
	 * the result of this is an empty set, then the processor must search for
	 * the provided text across all languages. The enumerations must be agreed
	 * upon by the query provider and processor. If the process does not
	 * recognize an enumeration, it must throw an
	 * <code>InvalidCriteriaException</code>.
	 * 
	 * @return a set of enumerations representing the languages to be searched
	 */
	public Set<Enum<?>> getLanguages();

	/**
	 * Retrieves the fields on which the text search should be performed. If
	 * this result of this is an empty set, then the processor must search for
	 * the provided text across all fields. The enumerations must be agreed upon
	 * by the query provider and processor. If the process does not recognize an
	 * enumeration, it must throw an <code>InvalidCriteriaException</code>.
	 * 
	 * @return a set of enumerations representing the fields to be searched
	 */
	public Set<Enum<?>> getSearchFields();

	/**
	 * Retrieves the text search string. The interpretation of this is to be
	 * agreed upon by the query provider and processor.
	 * 
	 * @return the search text
	 */
	public String getSearchText();

}
