package org.openmetadata.store.query;

import java.util.Set;

import org.openmetadata.store.exceptions.InvalidCriteriaException;

/**
 * Provides for querying of an object by the value of one of its properties. The
 * enumeration for the property field must be agreed upon by the provider and
 * processor of the query. If an unrecognized enumeration is used, then the
 * processor must throw a {@link InvalidCriteriaException}.
 * <p>
 * The set of values provided are <code>OR</code> joined. In order for a
 * property's value to match this search, its value must match one of the
 * provided {@link PropertyValue}s. Multiple values are typically only used to
 * specify value sets (e.g. <code>PROP_X = "A" OR "B")</code>.
 * 
 * @author Jack Gager
 * 
 */
public interface PropertySearch {

	/**
	 * Retrieves the property to which this search applies. The enumeration must
	 * be agreed upon by the query provider and processor. If the process does
	 * not recognize an enumeration, it must throw an
	 * <code>InvalidCriteriaException</code>.
	 * 
	 * @return an enumeration designating the property to be searched
	 */
	public Enum<?> getProperty();

	/**
	 * Retrieves the property values which constitute a match. This set is
	 * <code>OR</code> joined. An object is a match if the identified property's
	 * value matches any of the values supplied in this set.
	 * 
	 * @return a set of <code>PropertyValue</code>s that the searched property
	 *         must match
	 */
	public Set<PropertyValue> getValues();

}
