package org.openmetadata.store.query;

import java.util.List;
import java.util.Set;

import org.openmetadata.beans.IdentifiableBean;
import org.openmetadata.beans.paging.Paginator;
import org.openmetadata.store.exceptions.InvalidCriteriaException;

/**
 * Provides search criteria for querying {@link IdentifiableBean}s.
 * <p>
 * If a query is meant to be performed across multiple types of
 * <code>IdentifiableBean</code>s, then the parameter <code>B</code> should be
 * the superclass to which all objects to be searched are derived. All specified
 * parameters must be met for a match.
 * 
 * @author Jack Gager
 * 
 * @param <B>
 *            the base class of <code>IdentifiableBean</code> which the query
 *            applies.
 */
public interface Criteria<B extends IdentifiableBean> {
	
	/**
	 * Specifies the base type of <code>IdentifiableBean</code> for which the
	 * criteria applies.
	 * 
	 * @return the type of <code>IdentifiableBean</code> as its
	 *         <code>Class</code>.
	 */
	public Class<B> getBeanClass();
	
	/**
	 * Specifies all types of <code>IdentifiableBeans</code> for which the
	 * criteria applies. This is intended to be used by the query processor to
	 * determine how to search. This must contain the value from {@link #getBeanClass()}.
	 * 
	 * @return an array of the type of <code>IdentifiableBeans</code> as their
	 *         <code>Classes</code>.
	 */
	public Set<Class<? extends B>> getBeanClasses();

	/**
	 * Gets the number of items per page to be returned in the resulting
	 * {@link Paginator}. If the value is less than or equal to 0, the query
	 * processor may apply any value it wishes for the page size.
	 * 
	 * @return the number of items per page
	 */
	public int getPageSize();

	/**
	 * Gets the maximum number of items to be returned in the resulting
	 * <code>Paginator</code>. If the value is less than or equal to zero, the
	 * query processor may apply any limit is wishes.
	 * {@link Paginator#isComplete()} will indicate whether the result set is
	 * complete or not. A value of {@link Integer#MAX_VALUE} indicates that all
	 * results are requested to be returned, although the query processor may
	 * still impose its own limit.
	 * 
	 * @return the maximum number of items to be returned in the query result
	 */
	public int getMaxCount();

	/**
	 * Returns a collection of fields with a sort order that is to be applied to
	 * the result.
	 * 
	 * @return a collection of sorting details
	 */
	public List<Sort> getSort();

	public boolean searchParentContexts();

	public boolean searchChildContexts();

	/**
	 * Used to query for <code>IdentifiableBeans</code>s based on the
	 * identification of the <code>IdentifiableBean</code>s which they
	 * reference.
	 * <p>
	 * This contains a set of identifier sets. An identifier set (the inner set)
	 * is matched if any of the identifiers contained within it match (i.e. the
	 * identifiers are "OR" joined within the inner set). The entire set of
	 * identifier sets (the outer set) is matched if all identifier sets are
	 * matched (i.e the inner sets are "AND" joined within the outer set).
	 * <p>
	 * If the query processor does not handle the type of reference provided, it
	 * must throw an {@link InvalidCriteriaException}.
	 * <p>
	 * An empty set implies the search applies to all beans regardless of
	 * reference
	 * 
	 * @return a set of identifier sets for referenced objects
	 */
	public ReferenceSet getReferences();

	/**
	 * Used to query for <code>IdentifiableBeans</code>s based on the
	 * identification of <code>IdentifiableBean</code>s which reference them.
	 * <p>
	 * This contains a set of identifier sets. An identifier set (the inner set)
	 * is matched if any of the identifiers contained within it match (i.e. the
	 * identifiers are "OR" joined within the inner set). The entire set of
	 * identifier sets (the outer set) is matched if all identifier sets are
	 * matched (i.e the inner sets are "AND" joined within the outer set).
	 * <p>
	 * If the query processor does not handle the type of reference provided, it
	 * must throw an {@link InvalidCriteriaException}.
	 * <p>
	 * An empty set implies the search applies to all beans regardless of
	 * referrer
	 * 
	 * @return a set of identifier sets for referenced objects
	 */
	public ReferenceSet getReferrers();

	/**
	 * Indicates whether search text is provided. If this returns
	 * <code>true</code> {@link #getSearchText()} must not return
	 * <code>null</code>.
	 * 
	 * @return true if search text is provided otherwise, false.
	 */
	public boolean searchText();

	/**
	 * Used to query for <code>IdentifiableBeans</code>s based on text values.
	 * {@link #searchText()} should be called before this to ensure a text
	 * search applies.
	 * 
	 * @return search text or <code>null</code> if not applicable
	 */
	public TextSearch getSearchText();

	/**
	 * Used to query for <code>IdentifiableBeans</code>s based on property
	 * values. {@link #searchProperties()} should be called before this to
	 * ensure a property search applies. Multiple <code>PropertySearch</code>s
	 * may be provided for a single property. As with all of the other
	 * <code>Criteria</code> parameters, these are "AND" joined. Therefore, one
	 * can use multiple <code>PropertySearch</code>s to search a specific
	 * property for a value within a range (e.g. 0<= PROP_X < 10).
	 * <p>
	 * An empty set implies the search applies to all beans regardless of
	 * property values
	 * 
	 * @return a set of search properties
	 */
	public Set<PropertySearch> getSearchProperties();

}
