package org.openmetadata.store.managers.impl;

import java.util.HashSet;

import org.openmetadata.beans.IdentifiableBean;
import org.openmetadata.beans.notification.ChangeEvent.Type;
import org.openmetadata.beans.notification.IdentifiableChangeEvent;
import org.openmetadata.store.change.ChangeSet;
import org.openmetadata.store.change.impl.ChangeSetImpl;
import org.openmetadata.store.managers.ChangeManager;

/**
 * TODO Complete documentation
 * 
 * This implementation assumes that the changes to contained beans will result
 * in events raised for the container beans, and therefore it will only capture
 * events for the contained bean.
 * 
 * @author Jack Gager
 * 
 */
public class ContextContainerChangeManager implements ChangeManager {
	
	protected final Type changeType;
	protected final String contextId;
	protected final HashSet<String> additions;
	protected final HashSet<String> deletions;
	protected final HashSet<String> updates;
	protected final HashSet<String> containerAdditions;
	protected final HashSet<String> containerDeletions;
	protected final HashSet<String> containerUpdates;

	public ContextContainerChangeManager(String contextId) {
		this (contextId, Type.UPDATE);
	}
	
	public ContextContainerChangeManager(String contextId, Type changeType) {
		this.contextId = contextId;
		this.changeType = changeType;
		additions = new HashSet<String>();
		deletions = new HashSet<String>();
		updates = new HashSet<String>();
		containerAdditions = new HashSet<String>();
		containerDeletions = new HashSet<String>();
		containerUpdates = new HashSet<String>();
	}

	@Override
	public ChangeSet<String> getUnsavedChanges() {
		ChangeSetImpl<String> changeSet = new ChangeSetImpl<String>();
		if (!additions.isEmpty() || !deletions.isEmpty() | !updates.isEmpty()) {
			changeSet.getUpdates().add(contextId);
		}
		return changeSet;
	}

	@Override
	public ChangeSet<String> getSaveSet(String id) {
		ChangeSetImpl<String> changeSet = new ChangeSetImpl<String>();
		changeSet.getAdditions().addAll(containerAdditions);
		changeSet.getDeletions().addAll(containerDeletions);
		changeSet.getUpdates().addAll(containerUpdates);
		return changeSet;
	}

	@Override
	public ChangeSet<String> getDiscardSet(String id) {
		ChangeSetImpl<String> changeSet = new ChangeSetImpl<String>();
		changeSet.getAdditions().addAll(additions);
		changeSet.getDeletions().addAll(deletions);
		changeSet.getUpdates().addAll(updates);
		return changeSet;
	}

	@Override
	public ChangeSet<String> getAllSaveItems(String id) {
		ChangeSetImpl<String> saveSet = new ChangeSetImpl<String>();
		saveSet.getAdditions().addAll(additions);
		saveSet.getDeletions().addAll(deletions);
		saveSet.getUpdates().addAll(updates);
		return saveSet;
	}

	@Override
	public ChangeSet<String> getAllDiscardItems(String id) {
		ChangeSetImpl<String> discardSet = new ChangeSetImpl<String>();
		discardSet.getAdditions().addAll(additions);
		discardSet.getDeletions().addAll(deletions);
		discardSet.getUpdates().addAll(updates);
		return discardSet;
	}

	@Override
	public void notifyChangeEvent(IdentifiableChangeEvent event) {
		IdentifiableBean bean = event.getBean();
		String id = event.getBean().getPrimaryIdentifier();
		boolean container = bean.getContainerIdentifier().equals(id);
		switch (event.getType()) {
		case DELETE:
			if (additions.contains(id)) {
				additions.remove(id);
			} else {
				deletions.add(id);
			}
			if (container) {
				if (containerAdditions.contains(id)) {
					containerAdditions.remove(id);
				} else {
					containerDeletions.add(id);
				}
			}
			break;
		case CREATE:
			if (updates.contains(id))
				throw new RuntimeException(
						"Object cannot be added after it has been updated.");
			if (deletions.contains(id))
				throw new RuntimeException(
						"Object cannot be added after it has been deleted.");
			additions.add(id);
			if (container) {
				containerAdditions.add(id);
			}
			break;
		case UPDATE:
			if (deletions.contains(id))
				throw new RuntimeException(
						"Object cannot be updated after it has been deleted.");
			if (!additions.contains(id)) {
				updates.add(id);
			}
			if (container) {
				if (!containerAdditions.contains(id)) {
					containerUpdates.add(id);
				}
			}else {
				String containerId = bean.getContainerIdentifier();
				if (!containerAdditions.contains(containerId)) {
					containerUpdates.add(containerId);
				}
			}
			break;
		}
	}

	@Override
	public void notifyDiscard(String id) {
		if (id.equals(contextId)) {
			clear();
		}
		additions.remove(id);
		deletions.remove(id);
		updates.remove(id);
		containerAdditions.remove(id);
		containerDeletions.remove(id);
		containerUpdates.remove(id);
	}

	@Override
	public void notifySave(String id) {
		if (!id.equals(contextId)) {
			throw new RuntimeException("Only the context can be saved.");
		}
		clear();
	}
	
	public Type getChangeType() {
		return changeType;
	}
	
	public boolean isEmpty() {
		return additions.isEmpty() && deletions.isEmpty() && updates.isEmpty();
	}
	
	protected final void clear() {
		additions.clear();
		updates.clear();
		deletions.clear();
		containerAdditions.clear();
		containerUpdates.clear();
		containerDeletions.clear();
	}

}
