package org.openmetadata.store.managers.impl;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Set;

import org.openmetadata.beans.notification.IdentifiableChangeEvent;
import org.openmetadata.beans.notification.IdentifiableChangeListener;
import org.openmetadata.store.managers.ChangeNotificationManager;

public class ChangeNotificationManagerImpl implements ChangeNotificationManager {

	private HashMap<String, Set<IdentifiableChangeListener>> listenerMap;
	private HashMap<IdentifiableChangeListener, Set<String>> reverseMap;
	private HashSet<IdentifiableChangeListener> delegates;

	public ChangeNotificationManagerImpl() {
		listenerMap = new HashMap<String, Set<IdentifiableChangeListener>>();
		reverseMap = new HashMap<IdentifiableChangeListener, Set<String>>();
		delegates = new HashSet<IdentifiableChangeListener>();
	}

	@Override
	public void addDelegateListener(IdentifiableChangeListener listener) {
		delegates.add(listener);
	}

	@Override
	public void addChangeListener(String id, IdentifiableChangeListener listener) {
		getListenerSet(id).add(listener);
		getRegistrationSet(listener).add(id);
	}

	@Override
	public void removeChangeListener(String id,
			IdentifiableChangeListener listener) {
		getListenerSet(id).remove(listener);
	}

	@Override
	public void removeChangeListener(IdentifiableChangeListener listener) {
		for (String id : getRegistrationSet(listener)) {
			removeChangeListener(id, listener);
		}
		delegates.remove(listener);
	}

	@Override
	public void notifyChangeEvent(IdentifiableChangeEvent event) {
		for (IdentifiableChangeListener listener : viewListenerSet(event
				.getBean().getPrimaryIdentifier())) {
			listener.notifyChangeEvent(event);
		}
		for (IdentifiableChangeListener listener : delegates) {
			listener.notifyChangeEvent(event);
		}
	}

	/**
	 * Used to get the listener set for manipulation purposes. If a set does not
	 * exist in the map, one will be created. Therfore, this should only be used
	 * when the intention is to manipulate the listener set.
	 * 
	 * @param id
	 *            the bean identifier for which a listener set should be
	 *            returned
	 * @return the backed listener set for the bean
	 */
	protected final Set<IdentifiableChangeListener> getListenerSet(String id) {
		Set<IdentifiableChangeListener> set;
		if (listenerMap.containsKey(id)) {
			set = listenerMap.get(id);
		} else {
			set = new HashSet<IdentifiableChangeListener>();
			listenerMap.put(id, set);
		}
		return set;
	}

	/**
	 * Used to get the listener set, or an empty set if one does not exist. This
	 * set is not backed, so can it can be manipulated as needed without
	 * changing the underlying set. If the set does not exist, any empty set
	 * will be returned, but will not be put in the map.
	 * 
	 * @param id
	 *            the bean identifier for which a listener set should be
	 *            returned
	 * @return a non-backed listener set for the bean
	 */
	protected final Set<IdentifiableChangeListener> viewListenerSet(String id) {
		HashSet<IdentifiableChangeListener> set = new HashSet<IdentifiableChangeListener>();
		if (listenerMap.containsKey(id)) {
			set.addAll(listenerMap.get(id));
		}
		return set;
	}

	protected final Set<String> getRegistrationSet(
			IdentifiableChangeListener listener) {
		Set<String> set;
		if (reverseMap.containsKey(listener)) {
			set = reverseMap.get(listener);
		} else {
			set = new HashSet<String>();
			reverseMap.put(listener, set);
		}
		return set;
	}

}
