package org.openmetadata.store.managers;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class WorkspaceReferenceManager {

	private HashSet<String> deletedReferrers;
	private HashMap<String, Set<String>> referenceAdditions;
	private HashMap<String, Set<String>> referenceSubtractions;
	private HashMap<String, Set<String>> referrerAdditions;
	private HashMap<String, Set<String>> referrerSubtractions;
	protected Log logger = LogFactory.getLog(getClass());

	public WorkspaceReferenceManager() {
		deletedReferrers = new HashSet<String>();
		referenceAdditions = new HashMap<String, Set<String>>();
		referenceSubtractions = new HashMap<String, Set<String>>();
		referrerAdditions = new HashMap<String, Set<String>>();
		referrerSubtractions = new HashMap<String, Set<String>>();
	}

	public Set<String> getReferenceAdditions(String id) {
		return getSet(referenceAdditions, id);
	}

	public Set<String> getReferenceSubtractions(String id) {
		return getSet(referenceSubtractions, id);
	}

	public Set<String> getReferrerAdditions(String id) {
		return getSet(referrerAdditions, id);
	}

	public Set<String> getReferrerSubtractions(String id) {
		return getSet(referrerSubtractions, id);
	}

	public Set<String> getDeletedReferrers() {
		HashSet<String> returnSet = new HashSet<String>();
		returnSet.addAll(deletedReferrers);
		return returnSet;
	}

	public void addReferences(String fromId, String... toIds) {
		for (String toId : toIds) {
			logger.debug("Adding reference from: " + fromId + ", to: " + toId
					+ ".");
			Set<String> additions = new HashSet<String>();
			if (referenceAdditions.containsKey(fromId)) {
				additions = referenceAdditions.get(fromId);
			} else {
				referenceAdditions.put(fromId, additions);
			}
			additions.add(toId);

			additions = new HashSet<String>();
			if (referrerAdditions.containsKey(toId)) {
				additions = referrerAdditions.get(toId);
			} else {
				referrerAdditions.put(toId, additions);
			}
			additions.add(fromId);

			// If a reference is added, it should be removed from the
			// subtraction list, if applicable
			if (referenceSubtractions.containsKey(fromId)) {
				referenceSubtractions.get(fromId).remove(toId);
			}
			if (referrerSubtractions.containsKey(toId)) {
				referrerSubtractions.get(toId).remove(fromId);
			}
		}
	}

	public void removeReferences(String fromId, String... toIds) {
		for (String toId : toIds) {
			logger.debug("Removing reference from: " + fromId + ", to: " + toId
					+ ".");
			Set<String> subtractions = new HashSet<String>();
			if (referenceSubtractions.containsKey(fromId)) {
				subtractions = referenceSubtractions.get(fromId);
			} else {
				referenceSubtractions.put(fromId, subtractions);
			}
			subtractions.add(toId);

			subtractions = new HashSet<String>();
			if (referrerSubtractions.containsKey(toId)) {
				subtractions = referrerSubtractions.get(toId);
			} else {
				referrerSubtractions.put(toId, subtractions);
			}
			subtractions.add(fromId);

			// If a reference is subtracted, it should be removed from the
			// addition
			// list, if applicable
			if (referenceAdditions.containsKey(fromId)) {
				referenceAdditions.get(fromId).remove(toId);
			}
			if (referrerAdditions.containsKey(toId)) {
				referrerAdditions.get(toId).remove(fromId);
			}
		}
	}

	public void addDeletedReferrer(String id) {
		deletedReferrers.add(id);
	}

	public void clear(String id) {
		logger.debug("Clearing workspace reference information for: " + id
				+ ".");
		deletedReferrers.remove(id);
		if (referenceAdditions.containsKey(id)) {
			for (String to : referenceAdditions.get(id)) {
				if (referrerAdditions.containsKey(to)) {
					referrerAdditions.get(to).remove(id);
				}
			}
			referenceAdditions.remove(id);
		}
		if (referenceSubtractions.containsKey(id)) {
			for (String to : referenceSubtractions.get(id)) {
				if (referrerSubtractions.containsKey(to)) {
					referrerSubtractions.get(to).remove(id);
				}
				referenceSubtractions.remove(id);
			}
		}
	}
	
	public void clearAll() {
		logger.debug("Clearing all workspace reference information.");
		deletedReferrers.clear();
		referenceAdditions.clear();
		referenceSubtractions.clear();
		referrerAdditions.clear();
		referrerSubtractions.clear();
	}

	private HashSet<String> getSet(HashMap<String, Set<String>> map, String id) {
		HashSet<String> returnSet = new HashSet<String>();
		if (map.containsKey(id)) {
			returnSet.addAll(map.get(id));
		}
		return returnSet;
	}

}
