package org.openmetadata.store.managers;

import org.openmetadata.beans.IdentifiableBean;
import org.openmetadata.store.access.LockInformation;
import org.openmetadata.store.exceptions.ExistingLockException;
import org.openmetadata.store.exceptions.StoreException;

/**
 * This manages locks on shared instances of {@link IdentifiableBean}s for
 * editing purposes. This interacts with a shared repository space. An instance
 * of this is associated with a single user of the shared repository. Therefore,
 * the methods do not have user information associated with them.
 * 
 * @author Jack Gager
 * 
 */
public interface LockManager {

	/**
	 * Returns all active locks assigned by the manager instance. This will
	 * contain the locks assigned to the repository user associated with this
	 * manager.
	 * 
	 * @return active locks as an array of {@link LockInformation}
	 * 
	 * @see #getRepositoryLocks() getRepositoryLocks
	 */
	public LockInformation[] getLocks();

	/**
	 * Returns the lock information from the repository for the given
	 * <code>IdentifiableBean</code>.
	 * 
	 * @param id
	 *            the identifier of the <code>IdentifiableBean</code> for which
	 *            the lock information is sought
	 * @return the lock information for the <code>IdentifiableBean</code>
	 */
	public LockInformation getLockInformation(String id);

	/**
	 * Requests a lock for a given user on a particular
	 * <code>IdentifiableBean</code>. Locking a particular object may result in
	 * child objects being locked as well.
	 * 
	 * @param id
	 *            the identifier of the <code>IdentifiableBean</code> for which
	 *            the lock is sought
	 * @return the lock, if successful
	 * @throws ExistingLockException
	 *             if the requested object is already locked
	 * @throws StoreException
	 *             if the lock cannot be assigned to the
	 *             <code>IdentifiableBean</code>
	 */
	public LockInformation requestLock(String id) throws ExistingLockException,
			StoreException;

	// TODO The exceptions should be more explicit.
	/**
	 * Releases the lock on an <code>IdentifiableBean</code>. Only explicitly
	 * requested locks (i.e. <code>LockInformation</code> where the
	 * {@link LockInformation#hasParentLock() hasParentLock} method returns
	 * <code>false</code> may be released.
	 * 
	 * @param id
	 *            the identifier of the <code>IdentifiableBean</code> for which
	 *            the lock is to be released
	 * @throws StoreException
	 *             if the lock cannot be release due to an issue such as the
	 *             lock expiring
	 */
	public void releaseLock(String id) throws StoreException;

}
