package org.openmetadata.store.managers;

import org.openmetadata.beans.IdentifiableBean;
import org.openmetadata.beans.notification.IdentifiableChangeEvent;
import org.openmetadata.beans.notification.IdentifiableChangeListener;

/**
 * This manages listeners to changes of {@link IdentifiableBean}s.
 * Implementations must listen for changes and pass the received
 * {@link IdentifiableChangeEvent}s to all listeners registered for the changed
 * object(s).
 * <p>
 * This also applies to the originating events of a change, which is to say that
 * implementation must check whether there are listeners for the nested
 * initiating events for a received change notification.
 * 
 * @author Jack Gager
 * 
 */
public interface ChangeNotificationManager extends IdentifiableChangeListener {

	/**
	 * Adds a delegate listener, to which all received change event
	 * notifications will be passed.
	 * 
	 * @param listener
	 *            the <code>IdentifiableChangeListener</code> to be added as a
	 *            delegate
	 */
	public void addDelegateListener(IdentifiableChangeListener listener);

	/**
	 * Registers a <code>ChangeListner</code> for a particular
	 * <code>IdentifiableBean</code>.
	 * 
	 * @param id
	 *            the identifier of the <code>IdentifiableBean</code> for which
	 *            <code>IdentifiableChangeEvent</code>s should be passed to the
	 *            supplied <code>IdentifiableChangeListener</code>
	 * @param listener
	 *            the <code>IdentifiableChangeListener</code> to which received
	 *            <code>IdentifiableChangeEvent</code>s should be passed
	 */
	public void addChangeListener(String id, IdentifiableChangeListener listener);

	/**
	 * Unregisters a <code>IdentifiableChangeListener</code> for a particular
	 * <code>IdentifiableBean</code>.
	 * 
	 * @param id
	 *            the identifier of the <code>IdentifiableBean</code> for
	 *            which the <code>IdentifiableChangeListener</code> should be
	 *            removed
	 * @param listener
	 *            the <code>IdentifiableChangeListener</code> to be unregistered
	 */
	public void removeChangeListener(String id,
			IdentifiableChangeListener listener);

	/**
	 * A convenience method to remove the registrations for all objects for the
	 * given <code>IdentifiableChangeListener</code>. This is also used to
	 * remove delegate listeners.
	 * 
	 * @param listener
	 *            the <code>IdentifiableChangeListener</code> to be unregistered
	 */
	public void removeChangeListener(IdentifiableChangeListener listener);

}
