package org.openmetadata.store.managers;

import org.openmetadata.beans.notification.IdentifiableChangeListener;
import org.openmetadata.store.change.ChangeSet;

/**
 * This manager acts as a change listener and delegate notifier. Implementations
 * must retain the identifiers of all objects which report change events and
 * return these in the appropriate set according to the action of the event when
 * requested.
 * <p>
 * These sets are maintained by users of this class by the
 * <code>notifySave</code> and <code>notifyDiscard</code> methods.
 * <p>
 * Implementation must return all sets of identifiers as non-backed sets so that
 * any changes made by users of this class are not retained by the class itself.
 * <p>
 * Objects may exist in multiple lists, but the following priorities apply:
 * <ol>
 * <li>Deleted: a <code>DELETE</code> event will result in the object remaining
 * in the deleted list until it is saved or discarded, regardless of subsequent
 * events. If the object was previously in the added list, it should be removed
 * and not added to the deleted list. If the object was previously in the
 * updated list, it should remain there in case both the deletion and the prior
 * changes are reverted.</li>
 * <li>Added: a <code>CREATE</code> event will result in object remaining in the
 * added list until it is saved or discarded. It may be discarded by an explicit
 * call to the <code>notifyDiscard</code> method or by a subsequent
 * <code>DELETE</code> event. <code>UPDATE</code> events for added objects can
 * be ignored. A <code>CREATE</code> event must not occur after a
 * <code>UPDATE</code> or <code>DELETE</code> event or an unchecked exception
 * may be thrown.</li>
 * <li>Changed: a <code>UPDATE</code> event will result in the object being
 * placed in the changed list so long as it does not already exist in the added
 * or deleted lists. Objects will remain in this list until they are saved or
 * discarded. Objects may be discarded by an explicit call to the
 * <code>notifyDiscard</code> method or by a subsequent <code>DELETE</code>
 * event. An <code>UPDATE</code> event must not be preceded by a
 * <code>DELETE</code> event or an unchecked exception may be thrown.</li>
 * </ol>
 * 
 * @author Jack Gager
 * 
 */
public interface ChangeManager extends IdentifiableChangeListener {

	/**
	 * Returns the set of changes that can actually be saved. Note that this
	 * list may only be a subset of the actual objects that have changed and
	 * need to be saved which are returned by the {@link #getAdditions()},
	 * {@link #getDeletions()}, and {@link #getUpdates()} methods.
	 * 
	 * @return a {@link ChangeSet} with the identifiers of the changes which can
	 *         be explicitly saved
	 */
	public ChangeSet<String> getUnsavedChanges();

	/**
	 * Notifies the manager that the change, addition, or deletion of the object
	 * identified by the <code>id</code> parameter has been persisted and that
	 * the identifier can be removed from the appropriate set.
	 * 
	 * @param id
	 *            the identifier of the object for which the addition, change,
	 *            or deletion has been persisted.
	 */
	public void notifySave(String id);

	/**
	 * Notifies the manager that the change, addition, or deletion of the object
	 * identified by the <code>id</code> parameter is being discarded and that
	 * the identifier can be removed from the appropriate set.
	 * 
	 * @param id
	 *            the identifier of the object for which the addition, change,
	 *            or deletion is being discarded.
	 */
	public void notifyDiscard(String id);

	/**
	 * Returns the set of changes that should be persisted for a save operation
	 * on a given object.
	 * 
	 * @param id
	 *            the identifier of the object to be saved; this must be one of
	 *            the identifiers returned from the {@link #getUnsavedChanges()}
	 *            methods
	 * @return a {@link ChangeSet} with the identifiers of the objects which
	 *         must be saved as part of the save operation
	 */
	public ChangeSet<String> getSaveSet(String id);

	public ChangeSet<String> getDiscardSet(String id);	
	
	public ChangeSet<String> getAllSaveItems(String id);

	public ChangeSet<String> getAllDiscardItems(String id);

}
