package org.openmetadata.store.impl;

import java.util.ArrayList;
import java.util.Set;

import org.openmetadata.beans.IdentifiableBean;
import org.openmetadata.beans.deserialization.MutableDeserializer;
import org.openmetadata.beans.factory.BeanFactory;
import org.openmetadata.beans.notification.ChangeEvent.Type;
import org.openmetadata.beans.notification.IdentifiableChangeEvent;
import org.openmetadata.beans.notification.IdentifiableChangeListener;
import org.openmetadata.beans.serialization.Serializer;
import org.openmetadata.store.Workspace;
import org.openmetadata.store.access.AccessRights;
import org.openmetadata.store.cache.BeanCache;
import org.openmetadata.store.cache.LockingBeanCache;
import org.openmetadata.store.change.ChangeSet;
import org.openmetadata.store.change.impl.ChangeSetImpl;
import org.openmetadata.store.exceptions.InsufficientBeanRightsException;
import org.openmetadata.store.exceptions.InsufficientRightsException;
import org.openmetadata.store.exceptions.ObjectNotFoundException;
import org.openmetadata.store.exceptions.StoreException;
import org.openmetadata.store.managers.AccessManager;
import org.openmetadata.store.managers.ChangeManager;
import org.openmetadata.store.managers.ChangeNotificationManager;
import org.openmetadata.store.managers.WorkspaceReferenceManager;
import org.openmetadata.store.managers.impl.ChangeNotificationManagerImpl;
import org.openmetadata.store.repository.StoreRepository;
import org.openmetadata.store.repository.WorkspaceRepository;
import org.openmetadata.store.repository.notification.SaveEvent;
import org.openmetadata.store.repository.notification.SaveListener;

/**
 * 
 * @author Jack Gager
 * 
 */
public class WorkspaceImpl<Source extends Object> extends StoreImpl<Source>
		implements Workspace, IdentifiableChangeListener, SaveListener {

	private LockingBeanCache lockingBeanCache;
	private ChangeManager changeManager;
	private WorkspaceRepository<Source> workspaceRepository;
	private MutableDeserializer<Source> mutableDeserializer;
	private BeanFactory beanFactory;
	private ChangeNotificationManager changeNotificationManager;
	private WorkspaceReferenceManager localReferenceManager;
	private ArrayList<String> saveOperationList;

	public WorkspaceImpl(String contextId, boolean mustDeserialize) {
		super(contextId, true);
		changeNotificationManager = new ChangeNotificationManagerImpl();
		localReferenceManager = new WorkspaceReferenceManager();
		saveOperationList = new ArrayList<String>();
	}

	@Override
	public <B extends IdentifiableBean> B createBean(Class<B> beanClass)
			throws InsufficientRightsException {
		logger.debug("Creating bean of type: " + beanClass.getCanonicalName()
				+ ".");
		AccessRights rights = getRights(beanClass);
		if (!rights.canCreate()) {
			logger.debug("Could not create bean of type: "
					+ beanClass.getCanonicalName() + ".");
			throw new InsufficientRightsException(rights);
		}
		B bean = getBeanFactory().newInstance(beanClass);
		logger.debug("Created bean with id: " + bean.getPrimaryIdentifier()
				+ ".");
		return bean;
	}

	@Override
	public void discardAllChanges() {
		discardChanges(getContextId());
	}

	@Override
	public void discardChanges(String id) {
		logger.debug("Discarding changes for bean: " + id + ".");
		LockingBeanCache cache = getLockingBeanCache();
		ChangeManager changeManager = getChangeManager();
		ChangeSet<String> discardSet = changeManager.getDiscardSet(id);
		for (String u : discardSet.getUpdates()) {
			revertBean(u);
		}
		for (String d : discardSet.getDeletions()) {
			revertBean(d);
		}
		for (String a : changeManager.getAllDiscardItems(id).getAdditions()) {
			localReferenceManager.clear(a);
			if (cache.contains(a)) {
				cache.release(cache.get(a), this);
			}
		}
		for (String d : changeManager.getAllDiscardItems(id).getDeletions()) {
			localReferenceManager.clear(d);
			if (cache.contains(d)) {
				cache.release(cache.get(d), this);
			}
		}
		for (String u : changeManager.getAllDiscardItems(id).getUpdates()) {
			localReferenceManager.clear(u);
			if (cache.contains(u)) {
				cache.release(cache.get(u), this);
			}
		}
		changeManager.notifyDiscard(id);
	}

	@Override
	public ChangeNotificationManager getChangeNotificationManager() {
		logger.debug("Getting change notification manager.");
		return changeNotificationManager;
	}

	@Override
	public AccessRights getRights(IdentifiableBean bean) {
		logger.debug("Getting access right to bean: "
				+ bean.getPrimaryIdentifier() + ".");
		return getAccessManager().getRights(bean);
	}

	@Override
	public AccessRights getRights(Class<? extends IdentifiableBean> beanClass) {
		logger.debug("Getting access right to bean class: "
				+ beanClass.getCanonicalName() + ".");
		return getAccessManager().getRights(beanClass);
	}

	@Override
	public ChangeSet<String> getUnsavedChanges() {
		return getChangeManager().getUnsavedChanges();
	}

	@Override
	public synchronized void saveAllChanges()
			throws InsufficientBeanRightsException {
		saveChanges(getContextId());
	}

	@Override
	public synchronized void saveChanges(String id)
			throws InsufficientBeanRightsException {
		logger.debug("Saving changes for bean: " + id + ".");
		LockingBeanCache cache = getLockingBeanCache();
		ChangeManager changeManager = getChangeManager();
		ChangeSet<String> saveSet = changeManager.getSaveSet(id);
		ChangeSetImpl<Source> changeSet = new ChangeSetImpl<Source>();
		AccessManager accessManager = getAccessManager();
		for (String a : saveSet.getAdditions()) {
			IdentifiableBean bean = cache.get(a);
			if (accessManager.getRights(bean.getBeanType()).canCreate()) {
				changeSet.getAdditions().add(serializeBean(bean));
			}else {
				throw new InsufficientBeanRightsException(accessManager.getRights(bean.getBeanType()), bean);
			}
		}
		for (String u : saveSet.getUpdates()) {
			IdentifiableBean bean = cache.get(u);
			if (accessManager.getRights(bean).canEdit()) {
				changeSet.getUpdates().add(serializeBean(bean));
			}else {
				throw new InsufficientBeanRightsException(accessManager.getRights(bean.getBeanType()), bean);
			}
		}
		for (String d : saveSet.getDeletions()) {
			IdentifiableBean bean = cache.get(d);
			if (accessManager.getRights(bean).canDelete()) {
				changeSet.getDeletions().add(serializeBean(bean));
			}else {
				throw new InsufficientBeanRightsException(accessManager.getRights(bean.getBeanType()), bean);
			}
		}
		performSave(id, changeSet);
		for (String a : changeManager.getAllSaveItems(id).getAdditions()) {
			localReferenceManager.clear(a);
			if (cache.contains(a)) {
				cache.release(cache.get(a), this);
			}
		}
		for (String d : changeManager.getAllSaveItems(id).getDeletions()) {
			localReferenceManager.clear(d);
			if (cache.contains(d)) {
				cache.remove(cache.get(d));
			}
		}
		for (String u : changeManager.getAllSaveItems(id).getUpdates()) {
			localReferenceManager.clear(u);
			if (cache.contains(u)) {
				cache.release(cache.get(u), this);
			}
		}
		changeManager.notifySave(id);
	}

	@Override
	public void notifyChangeEvent(IdentifiableChangeEvent event) {
		logger.debug("Received change notification for bean: "
				+ event.getBean().getPrimaryIdentifier() + ".");
		processChangeEvent(event);
	}

	@Override
	public synchronized void notifySave(SaveEvent saveEvent) {
		LockingBeanCache cache = getLockingBeanCache();
		if (saveOperationList.contains(saveEvent.getEventIdentifier())) {
			saveOperationList.remove(saveEvent.getEventIdentifier());
			return;
		}
		for (String id : saveEvent.getPrimaryIdentifiers()) {
			if (cache.contains(id)) {
				// This will override any local changes
				revertBean(id);
				getChangeManager().notifyDiscard(id);
			}
		}
	}

	@Override
	public Set<String> getReferrers(String id) {
		logger.debug("Getting referrers for: " + id + ".");
		Set<String> referrers = super.getReferrers(id);
		referrers.addAll(localReferenceManager.getReferrerAdditions(id));
		referrers.removeAll(localReferenceManager.getReferrerSubtractions(id));
		referrers.removeAll(localReferenceManager.getDeletedReferrers());
		return referrers;
	}

	// TODO Integrate source query with query against the cache; it might make
	// sense to query the cache, and then passed the serialized hits to the
	// repository to be sorted against its hits. The result would then have to
	// account for repository hits that missed in the cache (which should be
	// removed); another approach could be to serialize all relevant updates and
	// search against the complete set of changes.

	@Override
	public void setBeanCache(BeanCache beanCache) {
		super.setBeanCache(beanCache);
		logger.warn("BeanCache must be set via the LockingBeanCache property. "
				+ "Cache set as BeanCache property will be overridden.");
	}

	public void setLockingBeanCache(LockingBeanCache lockingBeanCache) {
		if (this.lockingBeanCache != null
				&& !this.lockingBeanCache.equals(lockingBeanCache)) {
			throw raiseRuntimeException("LockingBeanCache cannot be reset.");
		}
		super.setBeanCache(lockingBeanCache);
		this.lockingBeanCache = lockingBeanCache;
	}

	@Override
	public void setStoreRepository(StoreRepository<Source> storeRepository) {
		super.setStoreRepository(storeRepository);
		logger.warn("StoreRepository must be set via the WorkspaceRespository property. "
				+ "Repository set as StoreRepository property will be overridden.");
	}

	public void setWorkspaceRepository(
			WorkspaceRepository<Source> workspaceRepository) {
		if (this.workspaceRepository != null
				&& !this.workspaceRepository.equals(workspaceRepository)) {
			throw raiseRuntimeException("WorkspaceRepository cannot be reset.");
		}
		super.setStoreRepository(workspaceRepository);
		workspaceRepository.addSaveListener(this);
		this.workspaceRepository = workspaceRepository;
		initializeBeanFactory(this.workspaceRepository.getBeanFactory());
		initializeMutableDeserializer(this.workspaceRepository.getDeserializer());
		initializeChangeManager(this.workspaceRepository.getChangeManager());
	}

	protected final LockingBeanCache getLockingBeanCache() {
		if (lockingBeanCache == null) {
			throw raiseRuntimeException("LockingBeanCache is not set.");
		}
		return lockingBeanCache;
	}

	protected final WorkspaceRepository<Source> getWorkspaceRepository() {
		if (workspaceRepository == null) {
			throw raiseRuntimeException("WorkspaceRepository is not set.");
		}
		return workspaceRepository;
	}

	protected final MutableDeserializer<Source> getMutableDeserializer() {
		if (mutableDeserializer == null
				&& getWorkspaceRepository().mustDeserialize()) {
			initializeMutableDeserializer(getWorkspaceRepository()
					.getDeserializer());
		}
		return mutableDeserializer;
	}

	protected final void initializeMutableDeserializer(
			MutableDeserializer<Source> mutableDeserializer) {
		if (mutableDeserializer == null) {
			throw raiseRuntimeException("MutableDeserializer cannot be null.");
		}
		if (this.mutableDeserializer != null
				&& !this.mutableDeserializer.equals(mutableDeserializer)) {
			throw raiseRuntimeException("MutableDeserializer cannot be reset.");
		}
		super.initializeDeserializer(mutableDeserializer);
		this.mutableDeserializer = mutableDeserializer;
		logger.debug("Setting workspace as change listener to the deserializer.");
		this.mutableDeserializer.setChangeListener(this);
	}

	protected final BeanFactory getBeanFactory() {
		if (beanFactory == null) {
			initializeBeanFactory(getWorkspaceRepository().getBeanFactory());
		}
		return beanFactory;
	}

	protected final void initializeBeanFactory(BeanFactory beanFactory) {
		if (beanFactory == null) {
			throw raiseRuntimeException("BeanFactory cannot be null.");
		}
		if (this.beanFactory != null && !this.beanFactory.equals(beanFactory)) {
			throw raiseRuntimeException("BeanFactory cannot be reset.");
		}
		this.beanFactory = beanFactory;
		logger.debug("Setting workspace as reseolver to the bean factory.");
		this.beanFactory.setResolver(this);
		logger.debug("Setting workspace as change listener to the bean factory.");
		this.beanFactory.setChangeListener(this);
	}

	protected final ChangeManager getChangeManager() {
		if (changeManager == null) {
			initializeChangeManager(getWorkspaceRepository()
					.getChangeManager());
		}
		return changeManager;
	}

	protected final void initializeChangeManager(ChangeManager changeManager) {
		if (this.changeManager != null
				&& !this.changeManager.equals(changeManager)) {
			throw raiseRuntimeException("ChangeManager cannot be reset.");
		}
		this.changeManager = changeManager;
	}

	protected synchronized void performSave(String contextId, ChangeSet<Source> sourceChangeSet)
			throws InsufficientBeanRightsException {
		logger.debug("Persisting beans.");
		try {
			String saveId = getWorkspaceRepository().save(contextId,
					sourceChangeSet.getAdditions(),
					sourceChangeSet.getUpdates(),
					sourceChangeSet.getDeletions());
			saveOperationList.add(saveId);
		} catch (InsufficientBeanRightsException bre) {
			throw bre;
		} catch (StoreException se) {
			throw raiseRuntimeException("Error saving changes.", se);
		}
	}
	
	protected final void addSaveOperation(String operationId) {
		saveOperationList.add(operationId);
	}

	protected void processChangeEvent(IdentifiableChangeEvent event) {
		LockingBeanCache cache = getLockingBeanCache();
		IdentifiableBean bean = event.getBean();
		// Verify bean is current, or cache it
		if (cache.contains(bean.getPrimaryIdentifier())) {
			logger.debug("Verifying bean with cache.");
			if (!cache.get(bean.getPrimaryIdentifier()).equals(bean)) {
				throw raiseRuntimeException("Change event bean does not match cached version of the same bean");
			}
		} else {
			logger.debug("Caching bean.");
			cache.add(bean);
		}
		logger.debug("Capturing bean.");
		cache.capture(bean, this);
		Type eventType = event.getType();
		logger.debug("Change notification type is: " + eventType + ".");
		if (eventType.equals(Type.DELETE)) {
			logger.debug("Updating references to reflect deletion.");
			localReferenceManager.addDeletedReferrer(bean
					.getPrimaryIdentifier());
		}
		logger.debug("Processing reference changes for change event");
		String fromId = event.getBean().getPrimaryIdentifier();
		localReferenceManager.addReferences(fromId, event.getNewReferences()
				.toArray(new String[] {}));
		localReferenceManager.removeReferences(fromId, event
				.getRemovedReferences().toArray(new String[] {}));
		logger.debug("Passing change notification to ChangeManager.");
		getChangeManager().notifyChangeEvent(event);
		getChangeNotificationManager().notifyChangeEvent(event);
	}

	protected Source serializeBean(IdentifiableBean bean) {
		logger.debug("Searializing bean: " + bean.getPrimaryIdentifier());
		if (getWorkspaceRepository().mustDeserialize()) {
			Serializer<Source> serializer = getWorkspaceRepository()
					.getSerializer();
			return serializer.serialize(serializer.getSourceClass(bean), bean);
		} else {
			@SuppressWarnings("unchecked")
			// Have to assume source class is
			// assignable from bean class
			Source source = (Source) bean;
			return source;
		}
	}

	protected void revertBean(String id) {
		LockingBeanCache cache = getLockingBeanCache();
		if (cache.contains(id)) {
			WorkspaceRepository<Source> repository = getWorkspaceRepository();
			if (repository.contains(id)) {
				try {
					IdentifiableBean bean = cache.get(id);
					getMutableDeserializer().deserialize(bean,
							repository.get(id));
					cache.release(bean, this);
				} catch (ObjectNotFoundException onfe) {
					throw raiseRuntimeException("Error reverting bean: " + id, onfe);
				}				
			}else {
				// Item was deleted
				cache.remove(cache.get(id));
			}
		}
		localReferenceManager.clear(id);
	}

}
