package org.openmetadata.store.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.openmetadata.beans.IdentifiableBean;
import org.openmetadata.beans.deserialization.Deserializer;
import org.openmetadata.beans.exceptions.ResolverException;
import org.openmetadata.beans.paging.Paginator;
import org.openmetadata.beans.paging.impl.PaginatorImpl;
import org.openmetadata.beans.reference.Resolver;
import org.openmetadata.store.Store;
import org.openmetadata.store.access.AccessRights;
import org.openmetadata.store.cache.BeanCache;
import org.openmetadata.store.exceptions.InsufficientRightsException;
import org.openmetadata.store.exceptions.InvalidCriteriaException;
import org.openmetadata.store.exceptions.ObjectNotFoundException;
import org.openmetadata.store.managers.AccessManager;
import org.openmetadata.store.query.Criteria;
import org.openmetadata.store.query.Result;
import org.openmetadata.store.query.SearchResult;
import org.openmetadata.store.query.impl.CriteriaImpl;
import org.openmetadata.store.repository.StoreRepository;

public class StoreImpl<Source extends Object> implements Store, Resolver {

	protected Log logger = LogFactory.getLog(getClass());
	private final String contextId;
	private BeanCache beanCache;
	private StoreRepository<Source> storeRepository;
	private Deserializer<Source> deserializer;
	private boolean isCaching;
	private final boolean mustCache;

	public StoreImpl(String contextId) {
		this(contextId, false);
	}

	public StoreImpl(String contextId, boolean mustCache) {
		this.contextId = contextId;
		this.mustCache = mustCache;
	}

	@Override
	public <B extends IdentifiableBean> B getBean(Class<B> beanClass, String id)
			throws ObjectNotFoundException, InsufficientRightsException {
		logger.debug("Getting bean of type: " + beanClass.getCanonicalName()
				+ ", with id: " + id + ".");
		B bean = _getBean(beanClass, id);
		AccessRights rights = getAccessManager().getRights(bean);
		if (!rights.canVeiw()) {
			logger.debug("User could not view requested object");
			throw new InsufficientRightsException(rights);
		}
		return bean;
	}

	@Override
	public <B extends IdentifiableBean> Paginator<B> query(Class<B> beanClass,
			Criteria<B> criteria) throws InvalidCriteriaException {
		// TODO Figure out access rights interaction
		SearchResult results = getStoreRepository().query(getContextId(),
				criteria);
		ArrayList<String> resultIds = new ArrayList<String>();
		for (Result result : results.getResults()) {
			resultIds.add(result.getIdentifier());
		}
		return new PaginatorImpl<B>(beanClass, this, criteria.getPageSize(),
				results.isComplete(), resultIds.toArray(new String[0]));
	}

	@Override
	public <B extends IdentifiableBean> B resolve(Class<B> beanClass, String id)
			throws ResolverException {
		logger.debug("Resolving bean: " + id + ".");
		try {
			return _getBean(beanClass, id);
		} catch (ObjectNotFoundException onfe) {
			throw new ResolverException(onfe.getPrimaryIdentifiers());
		}
	}

	@SuppressWarnings("unchecked")
	@Override
	public <B extends IdentifiableBean> Set<B> resolve(Class<B> beanClass,
			Set<String> ids) throws ResolverException {
		logger.debug("Resolving beans: " + ids + ".");
		HashSet<B> beanSet = new HashSet<B>();
		HashSet<String> nonCacheSet = new HashSet<String>();
		HashMap<String, B> beanMap = new HashMap<String, B>();
		for (String id : ids) {
			if ((mustCache || isCaching) && getBeanCache().contains(id)) {
				logger.debug("Found bean: " + id + ", in cache.");
				B bean = castBean(beanClass, getBeanCache().get(id));
				assert (bean != null);
				beanMap.put(bean.getPrimaryIdentifier(), bean);
			} else {
				nonCacheSet.add(id);
			}
		}
		logger.debug("Getting beans: " + nonCacheSet + ", from BeanRepository.");
		try {
			for (Source source : getStoreRepository().get(nonCacheSet)) {
				B bean = deserializeSource(beanClass, source);
				beanMap.put(bean.getPrimaryIdentifier(), bean);
				if (mustCache() || isCaching()) {
					bean = (B) getBeanCache().tryAdd(bean);
				}
			}
		} catch (ObjectNotFoundException onfe) {
			throw new ResolverException(onfe.getPrimaryIdentifiers());
		}
		logger.debug("Ordering results.");
		for (String id : ids) {
			if (beanMap.containsKey(id))
				beanSet.add(beanMap.get(id));
		}
		return beanSet;
	}

	@Override
	public Set<String> getReferrers(String id) {
		logger.debug("Retrieving referrers for: " + id + ".");
		CriteriaImpl<IdentifiableBean> criteria = new CriteriaImpl<IdentifiableBean>(
				IdentifiableBean.class);
		HashSet<String> references = new HashSet<String>();
		references.add(id);
		criteria.setReferencesDirect(true);
		criteria.addReference(references);
		try {
			SearchResult results = getStoreRepository().query(getContextId(),
					criteria);
			LinkedHashSet<String> resultIds = new LinkedHashSet<String>();
			for (Result result : results.getResults()) {
				resultIds.add(result.getIdentifier());
			}
			return resultIds;
		} catch (InvalidCriteriaException e) {
			throw raiseRuntimeException("Could not retrieve referrers.", e);
		}
	}

	public void setBeanCache(BeanCache beanCache) {
		if (this.beanCache != null && !this.beanCache.equals(beanCache)) {
			throw raiseRuntimeException("BeanCache cannot be reset.");
		}
		this.beanCache = beanCache;
		isCaching = true;
	}

	public void setStoreRepository(StoreRepository<Source> storeRepository) {
		if (this.storeRepository != null
				&& !this.storeRepository.equals(storeRepository)) {
			throw raiseRuntimeException("StoreRepository cannot be reset.");
		}
		this.storeRepository = storeRepository;
	}

	protected final AccessManager getAccessManager() {
		return getStoreRepository().getAccessManager();
	}

	protected final BeanCache getBeanCache() {
		if (beanCache == null) {
			throw raiseRuntimeException("BeanCache is not set.");
		}
		return beanCache;
	}

	protected final StoreRepository<Source> getStoreRepository() {
		if (storeRepository == null) {
			throw raiseRuntimeException("StoreRepository is not set.");
		}
		return storeRepository;
	}

	protected final Deserializer<Source> getDeserializer() {
		if (deserializer == null && getStoreRepository().mustDeserialize()) {
			initializeDeserializer(getStoreRepository().getDeserializer());
		}
		return deserializer;
	}

	protected final void initializeDeserializer(
			Deserializer<Source> deserializer) {
		if (deserializer == null) {
			throw raiseRuntimeException("Deserializer cannot be null.");
		}
		if (this.deserializer != null
				&& !this.deserializer.equals(deserializer)) {
			throw raiseRuntimeException("Deserializer cannot be reset.");
		}
		this.deserializer = deserializer;
		logger.debug("Settting Store as Resolver of Deserializer");
		deserializer.setResolver(this);
	}

	protected final String getContextId() {
		return contextId;
	}

	protected final boolean mustCache() {
		return mustCache;
	}

	protected final boolean isCaching() {
		return isCaching;
	}

	@SuppressWarnings("unchecked")
	protected <B extends IdentifiableBean> B _getBean(Class<B> beanClass,
			String id) throws ObjectNotFoundException {
		B bean;
		if ((mustCache() || isCaching()) && getBeanCache().contains(id)) {
			logger.debug("Returning object from cache.");
			bean = castBean(beanClass, getBeanCache().get(id));
		} else {
			logger.debug("Retrieving object from bean repository.");
			bean = deserializeSource(beanClass, getStoreRepository().get(id));
			if (mustCache() || isCaching()) {
				bean = (B) getBeanCache().tryAdd(bean);
			}
		}
		assert (bean != null);
		return bean;
	}

	protected IdentifiableBean getRootContainer(String id) {
		try {
			IdentifiableBean bean = _getBean(IdentifiableBean.class, id);
			if (bean.getContainerIdentifier().equals(
					bean.getPrimaryIdentifier())) {
				return bean;
			} else {
				return getRootContainer(bean.getContainerIdentifier());
			}
		} catch (ObjectNotFoundException e) {
			throw raiseRuntimeException("Could not find container bean.", e);
		}

	}

	protected final <B extends IdentifiableBean> B deserializeSource(
			Class<B> beanClass, Source source) {
		logger.debug("Deserializing source: " + source.toString());
		if (getStoreRepository().mustDeserialize()) {
			return getDeserializer().deserialize(beanClass, source);
		} else {
			return castBean(beanClass, source);
		}
	}

	protected final RuntimeException raiseRuntimeException(String message) {
		logger.error(message);
		return new RuntimeException(message);
	}

	protected final RuntimeException raiseRuntimeException(String message,
			Throwable exception) {
		logger.error(message, exception);
		return new RuntimeException(message, exception);
	}

	protected final <B extends IdentifiableBean> B castBean(Class<B> beanClass,
			Object bean) {
		if (!beanClass.isAssignableFrom(bean.getClass())) {
			throw raiseRuntimeException("The bean of type, "
					+ bean.getClass().getCanonicalName()
					+ ", cannot be converted to the type, "
					+ beanClass.getCanonicalName() + ".");
		}
		return beanClass.cast(bean);
	}

}
