package org.openmetadata.store.catalog.impl;

import java.util.Map;

import org.openmetadata.store.access.LockInformation;
import org.openmetadata.store.catalog.WorkspaceNode.ChangeState;
import org.openmetadata.store.change.ChangeSet;
import org.openmetadata.store.repository.ClientWorkspaceRepository;

public class CatalogHandler {

	public static <N extends ASharedWorkspaceNodeImpl<N, L>, L extends ASharedWorkspaceLevelImpl<N, L>> void processLevel(
			ASharedWorkspaceLevelImpl<N, L> level,
			Map<String, ? extends LockInformation> locks) {
		for (ASharedWorkspaceNodeImpl<N, L> node : level.getNodes()) {
			processNode(node, locks);
		}
		for (ASharedWorkspaceLevelImpl<N, L> child : level.getLevels()) {
			processLevel(child, locks);
		}
	}
	
	public static <N extends ASharedWorkspaceNodeImpl<N, L>, L extends ASharedWorkspaceLevelImpl<N, L>> void processNode(
			ASharedWorkspaceNodeImpl<N, L> node,
			Map<String, ? extends LockInformation> locks) {
		String id = node.getPrimaryIdentifier();
		node.setIsLocked(locks.containsKey(id), true);
		for (ASharedWorkspaceNodeImpl<N, L> child : node.getNodes()) {
			processNode(child, locks);
		}
		for (ASharedWorkspaceLevelImpl<N, L> level : node.getLevels()) {
			processLevel(level, locks);
		}
	}
	
	public static <N extends AClientWorkspaceNodeImpl<N, L>, L extends AClientWorkspaceLevelImpl<N, L>> void processLevel(
			AClientWorkspaceLevelImpl<N, L> level,
			ChangeSet<String> uncommittedChanges,
			Map<String, ? extends LockInformation> locks,
			ClientWorkspaceRepository<?> repository) {
		for (AClientWorkspaceNodeImpl<N, L> node : level.getNodes()) {
			processNode(node, uncommittedChanges, locks, repository);
		}
		for (AClientWorkspaceLevelImpl<N, L> child : level.getLevels()) {
			processLevel(child, uncommittedChanges, locks, repository);
		}
	}

	public static <N extends AClientWorkspaceNodeImpl<N, L>, L extends AClientWorkspaceLevelImpl<N, L>> void processNode(
			AClientWorkspaceNodeImpl<N, L> node,
			ChangeSet<String> uncommittedChanges,
			Map<String, ? extends LockInformation> locks,
			ClientWorkspaceRepository<?> repository) {
		String id = node.getPrimaryIdentifier();
		node.setIsShared(!uncommittedChanges.getAdditions().contains(id));
		node.setIsLocked(locks.containsKey(id), true);
		node.setIsAvailableOffline(repository.contains(id));
		if (uncommittedChanges.getAdditions().contains(id)) {
			node.setCommitState(ChangeState.ADDED);
		} else if (uncommittedChanges.getDeletions().contains(id)) {
			node.setCommitState(ChangeState.DELETED);
		} else if (uncommittedChanges.getUpdates().contains(id)) {
			node.setCommitState(ChangeState.UPDATED);
		} else {
			node.setCommitState(ChangeState.NONE);
		}
		for (AClientWorkspaceNodeImpl<N, L> child : node.getNodes()) {
			processNode(child, uncommittedChanges, locks, repository);
		}
		for (AClientWorkspaceLevelImpl<N, L> level : node.getLevels()) {
			processLevel(level, uncommittedChanges, locks, repository);
		}
	}

}
