package org.openmetadata.store.catalog;

import org.openmetadata.store.Workspace;
import org.openmetadata.text.ContextualTextSet;

/**
 * This defines the common features of a {@link Node} or {@link Level} in a
 * {@link Catalog}.
 * 
 * @author Jack Gager
 * 
 */
public interface CatalogItem {

	/**
	 * Indicates whether the node is a context object, that is an object from
	 * which a {@link Workspace} can be obtained.
	 * 
	 * @return <code>true</code> if the node is a context object
	 */
	public boolean isContext();

	/**
	 * Gets the identifier of context if {@link #isContext()} return
	 * <code>true</code>.
	 * 
	 * @return the identifier of the context object; <code>null</code> if the
	 *         item is not a context
	 */
	public String getContextId();

	/**
	 * Getter for the name(s) of the catalog item. The name may be provided in
	 * multiple values each for a unique context key.
	 * 
	 * @return the name as a ContextualTextSet
	 */
	public ContextualTextSet getName();

	/**
	 * Getter for the documentation(s) of the catalog item. Documentation may be
	 * provided in multiple values each for a unique context key.
	 * 
	 * @return the documentation as a ContextualTextSet
	 */
	public ContextualTextSet getDescription();

	/**
	 * Gets the collection of beans grouped in the item.
	 * 
	 * @return an array of child <code>Node</code>s
	 */
	public Node[] getNodes();

	/**
	 * Gets the collection of sub-groups for the item.
	 * 
	 * @return an array of child <code>Level</code>s
	 */
	public Level[] getLevels();

	/**
	 * Indicates whether this item is the child grouping for a bean
	 * <code>Node</code>.
	 * 
	 * @return <code>true</code> if the direct parent of this level is a
	 *         <code>Node</code>
	 */
	public boolean hasParentNode();

	/**
	 * Gets the bean <code>Node</code> in which this item is directly contained.
	 * This will return <code>null</code> if {@link #hasParentNode()
	 * hasParentNode} returns <code>false</code>.
	 * 
	 * @return the parent bean <code>Node</code>
	 */
	public Node getParentNode();

	/**
	 * Indicates whether this item is the child grouping for a higher grouping
	 * <code>Level</code>.
	 * 
	 * @return <code>true</code> if the direct parent of this level is another
	 *         <code>Level</code>
	 */
	public boolean hasParentLevel();

	/**
	 * Gets the grouping <code>Level</code> in which this item is directly
	 * contained. This will return <code>null</code> if
	 * {@link #hasParentLevel() hasParentLevel} returns <code>false</code>.
	 * 
	 * @return the parent grouping <code>Level</code>
	 */
	public Level getParentLevel();

}
