package org.openmetadata.store.access;

import java.util.Calendar;

import org.openmetadata.beans.IdentifiableBean;
import org.openmetadata.store.managers.LocalLockManager;

/**
 * Details lock information for an {@link IdentifiableBean} in a shared
 * repository. Locking a bean may result in associated beans being locked as
 * well, therefore this lock information can be hierarchical.
 * 
 * @author Jack Gager
 * 
 * @see LocalLockManager
 */
public interface LockInformation {

	/**
	 * Gets the primary identifier for the <code>IdentifiableBean</code> which
	 * this lock information pertains.
	 * 
	 * @return the identifier of the bean
	 */
	public String getPrimaryIdentifier();

	/**
	 * Indicates whether the <code>IdentifiableBean</code> is locked. If the
	 * bean is locked, it cannot be edited by anyone other than the user
	 * indicated in this lock information.
	 * 
	 * @return <code>true</code> if the object is locked, otherwise
	 *         <code>false</code>
	 */
	public boolean isLocked();

	/**
	 * Gets the repository user which owns this lock, assuming the object is
	 * locked. This must return a value if the {@link #isLocked() isLocked}
	 * method returns <code>true</code>.
	 * 
	 * @return the repository user which owns the lock, or <code>null</code> if
	 *         the object is not locked
	 */
	public User getUser();

	/**
	 * Gets the time at which the lock was granted, assuming {@link #isLocked()
	 * isLocked} method returns <code>true</code>.
	 * 
	 * @return the time at which the lock was granted.
	 */
	public Calendar getLockTime();

	/**
	 * Indicates whether the lock has an expiration time associated with it. If
	 * this is <code>false</code> then the {@link #getExpirationTime()} method
	 * will return <code>null</code>.
	 * 
	 * @return <code>true</code> is the lock has an expiration time, otherwise
	 *         <code>false</code>
	 */
	public boolean hasExpirationTime();

	/**
	 * Gets the time at which the lock has expired or will expire. This will
	 * return null if the lock has no expiration. Callers should check the
	 * {@link #hasExpirationTime()} method before calling this method.
	 * 
	 * @return the time at which the lock has expired or will expire, or
	 *         <code>null</code> if there is no expiration for the lock
	 */
	public Calendar getExpirationTime();

	/**
	 * Indicates the this lock information is contained within the context of
	 * another lock. This is only applicable is the {@link #isLocked() isLocked}
	 * method returns <code>true</code>.
	 * 
	 * @return <code>true</code> if this lock has a parent lock
	 */
	public boolean hasParentLock();

	/**
	 * Gets the parent lock in which this lock exists in the context. If the
	 * {@link #hasParentLock() hasParentLock} method returns <code>false</code>,
	 * this will return <code>null</code>. This is only applicable is the
	 * {@link #isLocked() isLocked} method returns <code>true</code>.
	 * 
	 * @return the parent lock details
	 */
	public LockInformation getParentLock();

	/**
	 * Gets the lock details for objects which are locked as a result of this
	 * locked. This is only applicable is the {@link #isLocked() isLocked}
	 * method returns <code>true</code>. The user identification for all child
	 * locks must be the same as this lock.
	 * 
	 * @return an array of lock details for all child objects locked as a result
	 *         of this lock
	 */
	public LockInformation[] getChildLocks();

}
