package org.openmetadata.store;

import org.openmetadata.beans.IdentifiableBean;
import org.openmetadata.store.access.AccessRights;
import org.openmetadata.store.change.ChangeSet;
import org.openmetadata.store.exceptions.BeanValidationExceptions;
import org.openmetadata.store.exceptions.InsufficientBeanRightsException;
import org.openmetadata.store.exceptions.InsufficientRightsException;
import org.openmetadata.store.managers.ChangeNotificationManager;
import org.openmetadata.store.managers.ContextManager;

/**
 * A generic store for any type of {@link IdentifiableBean} which allows for the
 * objects to be modified and will produce notifications of these changes. A
 * workspace has a single context bean associated with it, which is a higher
 * order bean in which other beans are defined and maintained.
 * <p>
 * Implementations must retain any modified or newly create bean for the life of
 * the workspace instance. Implementations must listen for change events from
 * the beans, and only reported changes are required to be recognized. TODO:
 * Update documentation from here down...
 * <p>
 * The <code>saveBean</code> method is only obligated to persist the modified or
 * created <code>Bean</code>s which have produced <code>ChangeEvent</code>
 * notifications.
 * <p>
 * Beans within the workspace may be managed within the context of other beans.
 * An object exists within the context of another object (referred to as a
 * context object) if saving the context object would result in saving the
 * object in question. The workspace exposes a {@link ContextManager} which
 * allows users to read and potentially update the context of the objects from
 * the store. The <code>ContextManager</code> also expresses which items from
 * the store can be saved via the {@link #saveChanges(String)} method.
 * Implementations must adhere to these contexts as follows;
 * <ol>
 * <li>Only objects which the context manager expressly allows to be saved
 * should be saved via the <code>saveChanges</code> method</li>
 * <li>Saving an object will result in saving of all objects within the context
 * of the object</li>
 * </ol>
 * <p>
 * It is assumed that each workspace instance is associated with a single user,
 * even if the workspace exposes data from a shared repository. In some cases,
 * it may be necessary to partition workspaces so that changes within a
 * workspace are isolated until such a time that they are saved to a shared
 * repository. In this case, a {@link WorkspaceProvider} may be used to retrieve
 * instances for a particular user (whether that user be an actual person or
 * some other distinct entity which needs a separate workspace).
 * 
 * @author Jack Gager
 * 
 */
public interface Workspace extends Store {

	/**
	 * Returns the {@link AccessRights} for a particular object in the
	 * workspace. Users of this workspace should always check the
	 * <code>AccessRights</code> before making changes to an object.
	 * 
	 * @param bean
	 *            the <code>IdentifaibleBean</code> for which the
	 *            <code>AccessRights</code> are sought
	 * @return the <code>AccessRights</code> for the object
	 */
	public AccessRights getRights(IdentifiableBean bean);

	/**
	 * Returns the {@link AccessRights} for a particular type of object in the
	 * workspace. These are general rights and may be superseded by the
	 * particular rights for an instance of the object type.
	 * 
	 * @param beanClass
	 *            the class of object for which the <code>AccessRights</code>
	 *            are sought
	 * @return the <code>AccessRights</code> for the object type available to
	 *         the user
	 */
	public AccessRights getRights(Class<? extends IdentifiableBean> beanClass);

	/**
	 * Creates a new instance of an object of the specified type. Users should
	 * check the {@link #getRights(Class) getRights} method to be sure they have
	 * the ability to create new instances of an object before calling this
	 * method.
	 * 
	 * @param <B>
	 *            a subclass of <code>Bean</Code>
	 * @param beanClass
	 *            the subclass of <code>Bean</code> which the created object
	 *            should be an instance of
	 * @return a new instance of the object type as the subclass specified in
	 *         the <code>beanClass</code> parameter
	 * @throws InsufficientRightsException
	 *             if there are not have sufficient rights to create the type of
	 *             object requested
	 */
	public <B extends IdentifiableBean> B createBean(Class<B> beanClass)
			throws InsufficientRightsException;

	/**
	 * Returns the list of object identifiers which have been changed but have
	 * not been saved.
	 * 
	 * TODO: Update documentation to reflect that this only contains identifiers
	 * that can actually be saved.
	 * 
	 * @return a <code>Set</code> of identifiers detailing the unsaved changes
	 *         (additions, updates, and deletions).
	 */
	public ChangeSet<String> getUnsavedChanges();

	/**
	 * TODO: Update all documentation TODO: Are exceptions needed here?
	 * 
	 * Saves the changes to the object in the store. Until this method is
	 * called, all changed objects are assumed to be working copies which could
	 * be reverted to the last saved state at any point. Saving an object will
	 * result in all objects that exist within the context of the object being
	 * saved as well. This information is available from the
	 * {@link #getContextManager() getContextmanager} method. Before calling
	 * this method, users should verify that the object can be explicitly saved
	 * via the <code>canSave</code> method of the <code>ContextManager</code>.
	 * Only <code>Bean</code>s created or retrieved directly from this store may
	 * be saved. Implementations must retain modified or created instances so
	 * that they can be saved when requested.
	 * 
	 * @param id
	 *            the identifier of the object to be saved
	 * 
	 * @throws InsufficientBeanRightsException
	 *             if there are not have sufficient permissions to update the
	 *             object or any of the modified objects within the context of
	 *             the saved object
	 * @throws BeanValidationExceptions
	 *             if a validation error occurs with any of the beans being
	 *             saved; these exceptions should be collected and thrown
	 *             together
	 */
	public void saveChanges(String id) throws InsufficientBeanRightsException,
			BeanValidationExceptions;

	public void saveAllChanges() throws InsufficientBeanRightsException,
			BeanValidationExceptions;

	/**
	 * TODO: Update all documentation
	 * 
	 * Discards the changes to the object in the store. Discarding changes to an
	 * object will result in all changes to objects that exist within the
	 * context of the object being discarded as well. This information is
	 * available from the {@link #getContextManager() getContextmanager} method.
	 * Unlike the {@link #saveChanges(String) saveChanges} method, it is not
	 * necessary for the requested object to be capable of being saved according
	 * to the <code>ContextManager</code> in order for changes to it to be
	 * discarded.
	 * <p>
	 * After calling this, any bean within the context of the identified bean
	 * must represent the last saved state of the bean. This includes any beans
	 * being actively used from the workspace. It is not necessary for users of
	 * a bean form which changes have been discarded to call the
	 * {@link #getBean(Class, String)} method to refresh the bean to the last
	 * persisted state.
	 * 
	 * @param id
	 *            the identifier of the object for which the changes are to
	 *            discarded
	 */
	public void discardChanges(String id);

	public void discardAllChanges();

	/**
	 * Retrieves the {@link ChangeNotificaitonManager} for the workspace. This
	 * allows for the addition and removal of listeners which can be notified of
	 * changes to the objects in the store.
	 * 
	 * @return the <code>ChangeNotificationManager</code> for the workspace
	 */
	public ChangeNotificationManager getChangeNotificationManager();

}
