package org.openmetadata.store;

import org.openmetadata.beans.IdentifiableBean;
import org.openmetadata.beans.paging.Paginator;
import org.openmetadata.store.exceptions.InsufficientRightsException;
import org.openmetadata.store.exceptions.InvalidCriteriaException;
import org.openmetadata.store.exceptions.ObjectNotFoundException;
import org.openmetadata.store.query.Criteria;

/**
 * The simplest form of a generic store for any bean which implements the
 * {@link IdentifiableBean} interface. The object represented by the bean can be
 * any type of information so long as it has a primary identifier which is
 * unique within the scope of this store. This interface only allows for
 * retrieval of the store objects.
 * 
 * @author Jack Gager
 * 
 */
public interface Store {

	/**
	 * Retrieves the bean with the supplied identifier from the store and
	 * returns it as the supplied subclass of <code>Bean</code>.
	 * 
	 * @param <B>
	 *            a subclass of <code>Bean</Code>
	 * @param beanClass
	 *            the subclass of <code>Bean</code> which the requested object
	 *            is to be returned as
	 * @param id
	 *            the identifier of the object requested
	 * @return the requested bean as the class provided in
	 *         <code>beanClass</code>
	 * @throws ObjectNotFoundException
	 *             if the object does not exist in the store
	 * @throws InsufficientRightsException
	 *             if there are not sufficient rights to view the object
	 */
	public <B extends IdentifiableBean> B getBean(Class<B> beanClass, String id)
			throws ObjectNotFoundException, InsufficientRightsException;

	/**
	 * Allows for querying of the store for objects which match the supplied
	 * criteria. Objects are returned in a {@link Paginator} which allows access
	 * to the beans in a paged manner.
	 * 
	 * @param <B>
	 *            a subclass of <code>IdentifiableBean</Code>
	 * @param beanClass
	 *            the subclass of <code>IdentifiableBean</code> which the queried objects
	 *            should be returned as in the <code>Paginator</code>
	 * @param criteria
	 *            the query parameters to which objects are to be matched
	 * @return a <code>Paginator</code> containing the objects which match the
	 *         supplied <code>Criteria</code>. This may be empty is the query
	 *         was valid but no matches resulted.
	 * @throws InvalidCriteriaException
	 *             if there is an issue with supplied <code>Criteria</code>
	 */
	public <B extends IdentifiableBean> Paginator<B> query(Class<B> beanClass,
			Criteria<B> criteria) throws InvalidCriteriaException;

}