package org.openmetadata.store;

import org.openmetadata.beans.IdentifiableBean;
import org.openmetadata.store.exceptions.BeanValidationExceptions;
import org.openmetadata.store.exceptions.ContextException;
import org.openmetadata.store.exceptions.InsufficientBeanRightsException;
import org.openmetadata.store.exceptions.InsufficientRightsException;
import org.openmetadata.store.exceptions.ObjectNotFoundException;
import org.openmetadata.store.snapshot.SnapshotHistory;

/**
 * A specialized {@link Workspace} which allows for the capturing and
 * restoration of save points for the beans exposed from the workspace. Note
 * that snapshots are only to be captured when the
 * {@link #saveChanges(String, String)} is invoked. The
 * {@link #saveChanges(String)} method can be used for saving changes without
 * capturing a restore point.
 * <p>
 * A snapshot only captures the state of the beans which were persisted in the
 * save operation. The state of all other beans are not captured as part of the
 * snapshot.
 * 
 * @author Jack Gager
 * 
 */
public interface SnapshotWorkspace extends Workspace {

	/**
	 * Performs the same function of the {@link #saveChanges(String)} method,
	 * but will also preserve a snapshot of the objects saved by the invocation.
	 * This snapshot will be part of the {@link SnapshotHistory} for this
	 * workspace.
	 * 
	 * @param id
	 *            the identifier of the object to be saved
	 * @param comment
	 *            used to capture a short comment about or a description of the
	 *            changes being saved
	 * @throws ContextException
	 *             if the object itself cannot be explicitly saved according to
	 *             the <code>ContextManager</code>
	 * @throws InsufficientBeanRightsException
	 *             if there are not have sufficient permissions to update the
	 *             object or any of the modified objects within the context of
	 *             the saved object
	 * @throws BeanValidationExceptions
	 *             if a validation error occurs with any of the beans being
	 *             saved; these exceptions should be collected and thrown
	 *             together
	 */
	public void saveChanges(String id, String comment) throws ContextException,
			InsufficientBeanRightsException, BeanValidationExceptions;

	/**
	 * Retrieves a snapshot version of a bean for the purpose of of viewing its
	 * contents. Any modifications to this bean will not be persisted in the
	 * workspace. In addition, retrieving a bean from this method should not
	 * result in the modification of any current or other snapshot versions of
	 * the same bean; that is to say that current and multiple snapshot versions
	 * must be simultaneously accessible.
	 * <p>
	 * In order to revert a bean to a snapshot version permanently, the
	 * {@link #restoreSnapshot(String, String...)} method must be used.
	 * 
	 * @param <B>
	 *            a subclass of <code>Bean</Code>
	 * @param beanClass
	 *            the subclass of <code>Bean</code> which the requested object
	 *            is to be returned as
	 * @param snapshotId
	 *            the identifier of the snapshot from which the identified bean
	 *            is to be retrieved
	 * @param id
	 *            the identifier of the object requested
	 * @return the requested bean as the class provided in
	 *         <code>beanClass</code>
	 * @throws ObjectNotFoundException
	 *             if the object does not exist in the store
	 * @throws InsufficientRightsException
	 *             if there are not sufficient rights to view the object
	 */
	public <B extends IdentifiableBean> B getSnapshotBean(Class<B> beanClass,
			String snapshotId, String id) throws ObjectNotFoundException,
			InsufficientRightsException;

	/**
	 * Restores the data from a snapshot to the specified beans, or if no bean
	 * identifiers are provided restores all changes from the identified
	 * snapshot.
	 * <p>
	 * After calling this method, all beans identified for the snapshot must
	 * represent the information that was saved at the point of the snapshot.
	 * This essentially means that the current version of any bean which is
	 * being restored must be updated. It is not necessary for a user of a
	 * restored bean to call the {@link #getBean(Class, String)} method to
	 * update any active beans to the restored snapshot.
	 * 
	 * @param snapshotId
	 *            the identifier of the snapshot from which the identified beans
	 *            are to be restored
	 * @param beanIds
	 *            identifiers of the beans from the snapshot which are to be
	 *            restored; if this is empty, all beans from the snapshot will
	 *            be restored
	 */
	public void restoreSnapshot(String snapshotId, String... beanIds);

	/**
	 * Retrieves snapshot information available for the workspace.
	 * 
	 * @return the available snapshots which can be restored
	 */
	public SnapshotHistory getSnapshotHistory();

}
