package org.openmetadata.dataset;

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;


public abstract class Transformation {

	Logger logger = Logger.getLogger(this.getClass().getName());
	
	//Name of the file used to create the datafile for the purpose of referencing it in newly created variables
	protected String dataFileName = "";

	//List of variables to be used during transformation, this will allow us to find the appropriate columns of data to transform
	protected Map<Variable, List<? extends Object>> inputVariables;

	//List of variables that are created from the transformation process. 
	protected Map<Variable, List<? extends Object>> outputVariables = new LinkedHashMap<Variable, List<? extends Object>>();

	/**
	 * Takes a single variable and a data source as parameters. 
	 * Adds the variable to the inputVariables as a key with its associated data as the value.
	 * Should only be used when only one variable is needed to perform the transformation.
	 * @param var
	 */
	public Transformation(Variable var, DataFile dataFile){
		//Retrieves datasources to pass the data values to the transformation
		DataSource dataSource = dataFile.getDataSource();

		//Retrieves the name of the file used to create the datafile in order to reference it in newly created variables
		dataFileName = dataFile.getFile().getName();

		inputVariables = new LinkedHashMap<Variable,List<? extends Object>>();
		try {
			inputVariables.put(var, dataSource.getValues(var));
		} catch (Exception e) {
			logger.log(Level.WARNING, var.getName()+" does not have any data associated with it.");
			inputVariables.put(var, null);
		}
	}

	/**
	 * Takes a list of variable objects and a data source as parameters.
	 * Adds each variable to the inputVariables as a key with their associated data as the value.
	 * Should only be used when more than one variable is needed to perform the transformation. 
	 * @param variableList
	 */
	public Transformation(List<Variable> variableList, DataFile dataFile){
		//Retrieves datasources to pass the data values to the transformation
		DataSource dataSource = dataFile.getDataSource();

		//Retrieves the name of the file used to create the datafile in order to reference it in newly created variables
		dataFileName = dataFile.getFile().getName();

		inputVariables = new LinkedHashMap<Variable, List<? extends Object>>();
		for(Variable var : variableList){
			try {
				inputVariables.put(var, dataSource.getValues(var));
			} catch (Exception e) {
				logger.log(Level.WARNING, var.getName()+" does not have any data associated with it.");
				inputVariables.put(var, null);
			}
		}
	}

	/**
	 * Performs the transform for a particular Transformation. 
	 * This method should populate the results list. 
	 */
	public abstract void Transform();

	/**
	 * Returns the variables that are created through the transformation
	 * @return
	 */
	public Map<Variable, List<? extends Object>> getOutputVariables() {
		return outputVariables;
	}

}
