package org.openmetadata.dataset;

import java.io.IOException;
import java.util.ArrayList;

import org.openmetadata.progress.ProgressIndicator;


/**
 * A file with data
 * @author Pascal Heus (pascal.heus@gmail.com)
 *
 */
public interface DataSource extends Monitorable{
	
	public enum STORAGE_MODEL {BY_ROW, BY_COL};
	
	public enum PARSE {PARSE, NO_PARSE};
	
	/**`
	 * Retrieves  list of values for the specified records
	 * @param recno
	 * @return 
	 * @throws Exception 
	 */
	public ArrayList<? extends Object> getRecord(Long recno) throws Exception;

	/**
	 * Retrieves all records in the dataset
	 * @return
	 */
	public ArrayList<ArrayList<? extends Object>> getRecords();
	
	
	/**
	 * Retrieves the value for the specified variable for the specified record 
	 * @param varno
	 * @param recno
	 * @return the value as an Object or null is missing
	 * @throws Exception 
	 */
	public Object getValue(Long varno, Long recno) throws Exception;
	
	/**
	 * Retrieves the value for the specified variable for the specified record 
	 * @param varno
	 * @param recno
	 * @return the value as an Object or null is missing
	 */
	public Object getValue(Variable var, Long recno) throws Exception;

	/**
	 * Returns the value for the specified variable and record 
	 * @param varno
	 * @param recno
	 * @return the collection of value as Object
	 * @throws Exception 
	 */
	public ArrayList<? extends Object> getValues(Long varno) throws Exception;

	/**
	 * Returns the value for the specified variable and record 
	 * @param var
	 * @return the collection of value as Object
	 * @throws ArrayIndexOutOfBoundsException
	 * @throws Exception 
	 */
	public ArrayList<? extends Object> getValues(Variable var) throws Exception;
	
	
	/**
	 * Loads the data in memory 
	 * @throws IOException 
	 * @throws Exception 
	 */
	public void loadInMemory(STORAGE_MODEL model) throws IOException, Exception;
	
	/**
	 * Frees all the data in memory 
	 */
	public void unloadFromMemory();
	
	/**
	 * Returns an estimated of the number of bytes needed to store the data in memory
	 * 
	 * @return
	 */
	public long estimateInMemorySize();
	
	/**
	 * @return the storage mode for this data source
	 */
	public  STORAGE_MODEL getStorageModel();
	
	/**
	 * Returns the data value that will take up the greatest width for any given variable.
	 * The String passed to it should be the variable name. 
	 * @param name
	 * @return
	 */
	public String getMaxValue(String name);
	
	/**
	 * Provides a way of calculating the unweighted summary statistics of the data
	 * @throws Exception
	 */
	public void produceUnweightedSummaryStatistics(boolean runStats) throws Exception;
	
	public void setProgressIndicator(ProgressIndicator progressIndicator);
	
	public ProgressIndicator getProgressIndicator();
	
	/**
	 * Used for command line tools. Indicates that some message should be displayed to 
	 * communicate what records have been read.
	 * @return
	 */
	public void setShowProgress(boolean showProgress);
	
	/**
	 * 
	 * @return
	 */
	public boolean getShowProgress();

	/**
	 * Used to get the correct summary statistics options Id to locate them in each variable
	 * @param weightVariables
	 * @return SummaryStatisticsOptions used to create the weighted sumstats.
	 * @throws Exception
	 */
	public SummaryStatisticsOptions produceWeightedSummaryStatistics(ArrayList<Variable> weightVariables)	throws Exception;

	/**
	 * Used to get the correct summary statistics options Id to locate them in each variable
	 * @return SummaryStatisticsOptions used to create the unweighted sumstats. 
	 */
	public SummaryStatisticsOptions getUnweightedSumOptions();
}
