package org.openmetadata.dataset;

import java.io.File;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.openmetadata.fileformat.FileFormat;

public interface DataFile {
		
	/**
	 * In Stat Packages that carry information in headers the description is usually composed of 
	 * bits and pieces that are picked out of the header and combined together in a String. This 
	 * could typically be the package name, version, or any other information that may be useful
	 * in describing the file. 
	 * 
	 * @return Descriptive information for this file to be used as comments in generated source code
	 */
	public String getDescription();
	
	/**
	 * Stat Packages may contain very specific information about the package in their data files. 
	 * This may include (but is not limited to) the package name, version, high/low endian files, etc.
	 * This information can be stored in a map of Strings and will be used in the production of DDI
	 * Lifecycle XML files. 
	 * 
	 * @return Software specific proprietary attributes of the file. 
	 */
	public Map<String, String> getProprietaryAttributes();
	
	/** 
	 * Used to keep track of what file was used to create the particular data file. 
	 * 
	 * @return the name of the file excluding any path and including extension
	 */
	public String getName();
	
	/**
	 * Returns the primary file that was used to create the dataFile
	 * 
	 * @return A java.io.File object or null if not applicable
	 */
	public File getFile();
	
	/**
	 * @return Ordered map of variables composing this file
	 * @throws Exception 
	 */
	public ArrayList<Variable> getVariables() throws Exception;
	
	public Variable getVariable(String variableName);
			
	public Map<String, CodeList> getCodeLists();
	
	public CodeList getCodeList(String codeListId);
	
	public void removeCodeList(String codeListId);
	
	/**
	 * @return Ordered map of variables composing the file primary key or NULL 
	 */
	public LinkedHashMap<String,Variable> getPrimaryKey();
	
	/**
	 * @return List of filters to apply to this file
	 */
	public List<TransformDefinition> getTransformDefinitions();
	
	/**
	 * @return the DataSource attached to this file or null if not available
	 */
	public DataSource getDataSource();
	
	/**
	 * @return the File Format of the data file
	 */
	public FileFormat getFileFormat();
	
	/**
	 * @return Number of records in this file or null if unknown 
	 */
	public Long getRecordCount();
	

	public void produceUnweightedSumStats() throws Exception;
	
	/**
	 * @return Number of variables in this file 
	 */
	public Integer getVariableCount();

	/**
	 * checks to see if the DataFile has been initialized;
	 */
	public boolean isInitialized();

	/**
	 * This is used to produce weighted summary statistics for the data in the data file. This is not done automatically
	 * because the weights must be provided for variables to be weighted. Accepts and arrayList of Variable objects to 
	 * specify the weights. 
	 * @param weightVariables
	 * @return 
	 * @throws Exception
	 */
	public SummaryStatisticsOptions produceWeightedSumStats(ArrayList<Variable> weightVariables) throws Exception;

}
