package org.openmetadata.text.impl;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.Set;

import org.openmetadata.text.ContextKey;
import org.openmetadata.text.ContextualText;
import org.openmetadata.text.ContextualTextSet;

/**
 * This is an immutable implementation of a set of contextual text values. The
 * text values may be simple string or mixed content (XHTML) and are uniquely
 * identified by a key. The key is an enumeration and an object.
 * 
 * @author Jack Gager
 * @author Kyaw Soe
 * 
 * @see OpenMetadataContextKey
 */
public abstract class AbstractContextualTextSetImpl<T extends ContextualTextImpl>
		implements ContextualTextSet {

	private final Set<T> itemSet;
	private final T[] emptyTypeArray;

	/**
	 * Constructs a contextual text set with the supplied contextual text items.
	 * The order of the items is preserved, and the first item is assumed to be
	 * the default value which will be provided in the absence of any context.
	 * 
	 * @param texts
	 * 
	 * @throws IllegalArgumentException
	 *             if there are duplicate keys in the contextual texts to be
	 *             added
	 */
	public AbstractContextualTextSetImpl(T... items) {
		emptyTypeArray = Arrays.copyOf(items, 0);

		Set<ContextKeySet> keySets = new HashSet<ContextKeySet>();
		itemSet = new LinkedHashSet<T>();
		for (T item : items) {
			ContextKeySet keySet = item.getKeySet();
			if (keySets.contains(keySet)) {
				throw new IllegalArgumentException(
						"Cannot create a contextual text set with containing duplicate keys.");
			}
			keySets.add(keySet);
			itemSet.add(item);
		}
	}

	protected abstract T createNew(ContextKey<?>... keys);

	@Override
	public int size() {
		return itemSet.size();
	}

	@Override
	public String[] getContextTypes() {
		Set<String> contextTypes = new HashSet<String>();
		for (T item : itemSet) {
			ContextKeySet keySet = item.getKeySet();
			for (ContextKey<?> key : keySet.getKeys()) {
				contextTypes.add(key.getType());
			}
		}
		return contextTypes.toArray(new String[0]);
	}

	@Override
	public T getDefault() {
		return itemSet.isEmpty() ? null : itemSet.iterator().next();
	}

	/**
	 * Returns the default text value for a given key. This will first try to
	 * find an exact match for the key, and then it will search for the first
	 * partial match.
	 * 
	 * @param keys
	 *            the context keys for which the default text value is sought
	 * @return the text with an exact matching key or the first text with a
	 *         matching key, or null if there are no matches.
	 */
	protected T getDefault(ContextKey<?>... keys) {
		if (keys == null) {
			throw new IllegalArgumentException(
					"Cannot get a contextual text associated with null keys.");
		}

		ContextKeySet testSet = new ContextKeySet(keys);
		for (T item : itemSet) {
			if (item.getKeySet().equals(testSet)) {
				return item;
			}
		}
		return null;
	}

	@Override
	public T get(ContextKey<?>... keys) {
		T existingObj = this.getDefault(keys);
		if (existingObj != null) {
			return existingObj;
		} else {
			T newObj = this.createNew(keys);
			itemSet.add(newObj);
			return newObj;
		}
	};

	@Override
	public T[] find(ContextKey<?>... keys) {
		ArrayList<T> returnSet = new ArrayList<T>();
		for (T item : itemSet) {
			if (item.getKeySet().matches(keys)) {
				returnSet.add(item);
			}
		}
		return returnSet
				.toArray(Arrays.copyOf(emptyTypeArray, returnSet.size()));
	}

	@Override
	public ContextualText[] getAll() {
		return itemSet.toArray(Arrays.copyOf(emptyTypeArray, itemSet.size()));
	}

	@Override
	public boolean remove(ContextualText element) {
		return itemSet.remove(element);
	}

	@Override
	public boolean removeAll(Collection<? extends ContextualText> coll) {
		return itemSet.removeAll(coll);
	}

	@Override
	public boolean removeAll(ContextKey<?>... keys) {
		T[] matchedItems = this.find(keys);
		return itemSet.removeAll(Arrays.asList(matchedItems));
	};
}
