package org.openmetadata.cache.impl;

import java.lang.ref.ReferenceQueue;
import java.lang.ref.SoftReference;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.openmetadata.cache.Cache;

/**
 * This class wraps a map of soft references to identified objects. When an
 * object is captured, it is mapped so that there is a strong reference to the
 * captured object. This ensures that it will not be disposed. All other objects
 * may be freed up at the discretion of the garbage collector.
 * 
 * @author Jack Gager
 * 
 * @param <T>
 *            the base object held in this cache
 */
public abstract class SoftCache<T extends Object> implements Cache<T> {

	protected final Log logger = LogFactory.getLog(getClass());

	/** The internal HashMap that will hold the SoftReference. */
	protected final Map<String, SoftIdentifiableObjectReference> hash;

	/** Reference queue for cleared SoftReference objects. */
	private final SoftIdentifiableObjectReferenceQueue queue;

	/**
	 * ThreadLocal variable for holding the last checked object to be sure it
	 * isn't collected
	 */
	private ThreadLocal<T> lastChecked;

	/**
	 * ThreadLocal variable for holding the id of the last checked object so it
	 * can be quickly retrieved
	 */
	private ThreadLocal<String> lastCheckedId;

	public SoftCache() {
		hash = new HashMap<String, SoftIdentifiableObjectReference>();
		queue = new SoftIdentifiableObjectReferenceQueue();
		lastChecked = new ThreadLocal<T>();
		lastCheckedId = new ThreadLocal<String>();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.openmetadata.cache.IdentifiableObjectCache#contains(java.lang.String)
	 * 
	 * The SoftHashMap class will retain a single requested object for a given
	 * thread, therefore the get method can assured to return an object if this
	 * method returns true, so long as subsequent calls to this method are not
	 * made in the same thread
	 */
	@Override
	public boolean contains(String id) {
		logger.debug("Checking for object: " + id);
		if (id.equals(lastCheckedId.get())) {
			return true;
		}
		expungeStaleEntries();
		if (hash.containsKey(id)) {
			SoftIdentifiableObjectReference soft_ref = hash.get(id);
			if (soft_ref != null) {
				logger.debug("Found reference for: " + id);
				T result = soft_ref.get();
				if (result != null) {
					logger.debug("Capturing referent for: " + id);
					lastChecked.set(result);
					lastCheckedId.set(id);
					return true;
				} else {
					hash.remove(id);
				}
			} else {
				logger.debug("Removing null map entry for: " + id);
				hash.remove(id);
			}
		}
		return false;
	}

	@Override
	public T get(String id) {
		logger.debug("Getting object: " + id);
		// If this was checked last get it from the lastChecked
		if (id.equals(lastCheckedId.get())) {
			return lastChecked.get();
		}
		SoftIdentifiableObjectReference ref = hash.get(id);
		// Check that the entry wasn't expunged
		if (ref==null) {
			return null;
		}
		T obj = ref.get();
		if (obj == null) {
			// Object has been collected so expunge this entry
			hash.remove(id);
		}
		return obj;
	}

	@Override
	public void add(T object) {
		String id = extractId(object);
		if (id.equals(lastCheckedId.get())) {
			lastChecked.set(object);
		}
		logger.debug("Adding object: " + id);
		if (hash.containsKey(id)) {
			logger.debug("Updating existing object: " + id);
		}
		hash.put(id, new SoftIdentifiableObjectReference(object, queue));
	}

	@Override
	public void remove(T object) {
		String id = extractId(object);
		if (id.equals(lastCheckedId.get())) {
			lastCheckedId.remove();
			lastChecked.remove();
		}
		logger.debug("Removing object: " + id);
		hash.remove(id);
	};

	/**
	 * Extracts the unique identifier from the object so that it may be
	 * identified within the map
	 * 
	 * @param object
	 *            the object to be mapped
	 * @return the unique identifier for the object
	 */
	abstract protected String extractId(T object);

	protected final void expungeStaleEntries() {
		SoftIdentifiableObjectReference sv;
		while ((sv = queue.poll()) != null) {
			String id = sv.getId();
			logger.debug("Expunging entry for disposed object: " + id);
			hash.remove(id);
		}
	}

	protected final class SoftIdentifiableObjectReference extends
			SoftReference<T> {

		private final String id;

		private SoftIdentifiableObjectReference(T object,
				SoftIdentifiableObjectReferenceQueue queue) {
			super(object, queue);
			id = extractId(object);
		}

		private String getId() {
			return id;
		}

	}

	protected final class SoftIdentifiableObjectReferenceQueue extends
			ReferenceQueue<T> {

		@SuppressWarnings("unchecked")
		@Override
		public SoftIdentifiableObjectReference poll() {
			return (SoftIdentifiableObjectReference) super.poll();
		}

	}

}
