package org.openmetadata.cache.impl;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Set;

import org.openmetadata.cache.LockingCache;

/**
 * This class wraps a map of soft references to identified objects. When an
 * object is captured, it is mapped so that there is a strong reference to the
 * captured object. This ensures that it will not be disposed. All other objects
 * may be freed up at the discretion of the garbage collector.
 * 
 * @author Jack Gager
 * 
 * @param <T>
 *            the base object held in this cache
 */
public abstract class SoftLockingCache<T extends Object> extends SoftCache<T>
		implements LockingCache<T> {

	private HashMap<T, Set<Object>> captureMap;

	public SoftLockingCache() {
		super();
		captureMap = new HashMap<T, Set<Object>>();
	}

	@Override
	public void remove(T object) {
		logger.debug("Removing object");
		if (!captureMap.containsKey(object)) {
			super.remove(object);
		}
		logger.debug("Object cannot be removed");
	};

	@Override
	public void capture(T object, Object requester) {
		logger.debug("Capturing object");
		String id = extractId(object);
		// First, ensure that the bean is in the object map
		if (!hash.containsKey(id) || hash.get(id) == null
				|| hash.get(id).get() == null) {
			logger.warn("Unexpected forced capture of [" + id + "]");
			add(object);
		}
		Set<Object> requesterSet = captureMap.get(object);
		if (requesterSet == null) {
			requesterSet = new HashSet<Object>();
		}
		requesterSet.add(requester);
		captureMap.put(object, requesterSet);
	}

	@Override
	public void release(T object, Object requester) {
		logger.debug("Releasing object");
		Set<Object> requesterSet = captureMap.get(object);
		if (requesterSet == null) {
			captureMap.remove(object);
		} else {
			requesterSet.remove(requester);
			if (requesterSet.isEmpty()) {
				captureMap.remove(object);
			}
		}
	}

}
