package org.openmetadata.text;

import java.util.Collection;

/**
 * This is an immutable implementation of a set of contextual text values. The
 * text values may be simple string or mixed content (XHTML) and are uniquely
 * identified by a key. The key is an enumeration and an object.
 * 
 * @author Jack Gager
 * @author Kyaw
 * 
 * @see OpenMetadataContextKey
 */
public interface ContextualTextSet {

	/**
	 * Return the number of contextual texts in this set.
	 * 
	 * @return an integer reflecting the number of contextual texts.
	 */
	int size();

	/**
	 * Returns the unique context keys associated with the
	 * <code>ContextualString</code> objects in this set.
	 * 
	 * @return an array of <code>String</code> reflecting the all unique context
	 *         keys.
	 */
	String[] getContextTypes();

	/**
	 * Returns the default text value if the set is not empty. Null will be
	 * returned if there is no contextual text item in the set.
	 * 
	 * @return default contextual text
	 */
	ContextualText getDefault();

	/**
	 * Returns the text where the context keys are the exact match of the keys
	 * sought. If An exact match for the keys could not be found, and then a new
	 * <code>ContextualText</code> object will be created with the keys sought.
	 * 
	 * @param keys
	 *            the context keys for which the text value is sought. it must
	 *            not be <code>null</code>.
	 * @return an existing text with an exact matching key or the first text
	 *         with a matching key, or new text with the given keys if there are
	 *         no matches.
	 */
	ContextualText get(ContextKey<?>... keys);

	/**
	 * Gets the contextual text items matching the keys provided. This may
	 * return more than one contextual text if the provided key is a subset of
	 * the contexts of the actual text items. If the empty array of keys is
	 * provided, all the contextual text items in the set will be returned.
	 * 
	 * @param keys
	 *            the context keys for which text values are sought
	 * @return the context text items with full or partial keys matching the
	 *         provided key; array will be empty if there are no matches
	 */
	ContextualText[] find(ContextKey<?>... keys);

	/**
	 * Removes the specified contextual text element if it is contained in the
	 * set.
	 * 
	 * @param element
	 *            contextual text element to be removed from the set.
	 * @return true if this set changed as a result of the call.
	 */
	boolean remove(ContextualText element);

	/**
	 * Removes all contextual text elements that are contained in the specified
	 * collection.
	 * 
	 * @param coll
	 *            collection containing the contextual texts to be removed from
	 *            this set.
	 * @return true if this set changed as a result of the call.
	 */
	boolean removeAll(Collection<? extends ContextualText> coll);

	/**
	 * Removes all contextual text elements where the associated keys either
	 * exactly match or partially match the keys specified. If no key is
	 * specified, all the items in the set will be removed.
	 * 
	 * @param keys
	 *            the context keys.
	 * @return
	 */
	boolean removeAll(ContextKey<?>... keys);

	/**
	 * Gets all the contextual text objects.
	 * 
	 * @return an array of <code>ContextualText</code> objects.
	 */
	ContextualText[] getAll();

}
