package org.openmetadata.datacube.impl;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.openmetadata.datacube.DataCube;
import org.openmetadata.datacube.DimensionKey;

/**
 * Simple implementation of <code>DataCube</code> which represents a
 * multidimensional immutable dataset where the data items can have an arbitrary
 * number of dimensions.
 * 
 * @author Kyaw Soe
 * 
 * @param <T>
 */
public class DataCubeImpl<T extends DataItemImpl> implements DataCube<T> {

	/**
	 * All children items in this cube.
	 */
	private final T[] dataItems;

	/**
	 * All the distinct keys associated with the children data items in this
	 * cube.
	 */
	private final DimensionKey<?>[] distinctKeys;

	/**
	 * Dimensions associated with the data items.
	 */
	private final Set<String> dimensions;

	private final Set<Class<?>> dimensionClasses;

	/**
	 * Constructs a new <code>DataCube<code> object with the given data items.
	 * 
	 * @param dataItems
	 *            an array of <code>DataItem</code> objects.
	 */
	public DataCubeImpl(T... dataItems) {
		this.dataItems = dataItems;

		this.dimensions = new HashSet<String>();
		this.dimensionClasses = new HashSet<Class<?>>();

		Set<DimensionKey<?>> distinctKeySet = new HashSet<DimensionKey<?>>();

		Set<DimensionKeySet> keySets = new HashSet<DimensionKeySet>();
		for (T dataItem : dataItems) {
			DimensionKeySet keySet = dataItem.getDimensionKeySet();
			/*if (keySets.contains(keySet)) {
				throw new IllegalArgumentException(
						"Cannot create a data item set with containing duplicate keys.");
			}*/
			
			for(DimensionKeySet existingKeySet : keySets) {
				if(existingKeySet.equals(keySet)){
					throw new IllegalArgumentException(
							"Cannot create a data item set with containing duplicate keys.");
				}
			}
			
			keySets.add(keySet);

			for (DimensionKey<?> key : keySet.getKeys()) {
				distinctKeySet.add(key);

				dimensions.add(key.getType());
				dimensionClasses.add(key.getTypeClass());
			}
		}

		distinctKeys = distinctKeySet.toArray(new DimensionKey<?>[0]);
	}
	
	@Override
	public int size() {
		return dataItems.length;
	}

	@Override
	public String[] getDimensions() {
		return dimensions.toArray(new String[0]);
	}

	/**
	 * Return an array of
	 * <code>Class<code> objects reflecting the dimensions associated with the data items.
	 * 
	 * @return an array of <code>Class<code> objects representing the dimensions.
	 */
	public Class<?>[] getDimensionClasses() {
		return dimensionClasses.toArray(new Class<?>[0]);
	}

	/**
	 * Return all the distinct keys associated with the children data items in
	 * this cube.
	 * 
	 * @return an array containing the all distinct
	 *         <code>DimensionKey<code> objects
	 */
	public DimensionKey<?>[] getDistinctKeys() {
		return distinctKeys;
	}

	@Override
	public T[] getDataItems(DimensionKey<?>... dimensionKeys) {
		List<T> returnSet = new ArrayList<T>();
		for (T dataItem : this.dataItems) {
			if (dataItem.getDimensionKeySet().matches(dimensionKeys)) {
				returnSet.add(dataItem);
			}
		}
		return returnSet.toArray(Arrays.copyOf(dataItems, returnSet.size()));
	}
}
